/*
 * <one line to give the program's name and a brief idea of what it does.>
 * Copyright (C) 2018  camilo <email>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "placeslist.h"
#include "fm.h"
#include "tagging.h"

#include <QEventLoop>
#include <QFileSystemWatcher>
#include <QIcon>
#include <QTimer>

#include <MauiKit/Core/utils.h>
#include <MauiKit/Core/fmh.h>

#ifdef COMPONENT_ACCOUNTS
#include <MauiKit/Core/mauiaccounts.h>
#endif

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
#include <KFilePlacesModel>
#include <Solid/Device>
#endif

#if defined Q_OS_ANDROID || defined Q_OS_WIN32 || defined Q_OS_MACOS || defined Q_OS_IOS
PlacesList::PlacesList(QObject *parent)
    : MauiList(parent)
    , fm(new FM(this))
    , model(nullptr)
    , watcher(new QFileSystemWatcher(this))
    #else
PlacesList::PlacesList(QObject *parent)
    : MauiList(parent)
    , fm(new FM(this))
    , model(new KFilePlacesModel(this))
    , watcher(new QFileSystemWatcher(this))
    #endif
{
    /*
     *  The watcher signal returns a local file URL withouth a scheme, and the model is using a local file URL with file:// scheme.
     *  So those need to be correctly mapped
     * */
    connect(watcher, &QFileSystemWatcher::directoryChanged, [&](const QString &path) {
        if (this->count.contains(QUrl::fromLocalFile(path).toString())) {
            const auto oldCount = this->count[QUrl::fromLocalFile(path).toString()];
            const auto index = this->indexOf(FMH::MODEL_KEY::PATH, QUrl::fromLocalFile(path).toString());
            const QDir dir(path);
            const auto newCount = dir.count();
            int count = newCount - oldCount;

            this->list[index][FMH::MODEL_KEY::COUNT] = QString::number(std::max(0, count));
            emit this->updateModel(index, {FMH::MODEL_KEY::COUNT});
        }
    });

#ifdef COMPONENT_ACCOUNTS
    connect(MauiAccounts::instance(), &MauiAccounts::accountAdded, this, &PlacesList::setList);
    connect(MauiAccounts::instance(), &MauiAccounts::accountRemoved, this, &PlacesList::setList);
#endif

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
    
    connect(this->model, &KFilePlacesModel::reloaded, this, &PlacesList::setList);

    connect(this->model, &KFilePlacesModel::setupDone, this, &PlacesList::setList);
    
    connect(this->model, &KFilePlacesModel::rowsInserted, [this](const QModelIndex, int, int) {
        this->setList();
        emit this->bookmarksChanged();

        /*emit this->preListChanged();

        for (int i = first; i <= last; i++)
        {
            const QModelIndex index = model->index(i, 0);

            if(this->groups.contains(model->groupType(index)))
            {
                this->list << getGroup(*this->model, static_cast<FMH::PATHTYPE_KEY>(model->groupType(index)));
            }
        }
        emit this->postListChanged();	*/
    }); // TODO improve the usage of the model
#else
    connect(&AppSettings::global(), &AppSettings::settingChanged, [&](const QUrl, const QString &key, const QVariant, const QString &group) {
        if (key == "BOOKMARKS" && group == "PREFERENCES") {
            this->setList();
            emit this->bookmarksChanged();
        }
    });
#endif
}

void PlacesList::watchPath(const QString &path)
{
    if (path.isEmpty() || !FMH::fileExists(path) || !QUrl(path).isLocalFile())
        return;

    this->watcher->addPath(QUrl(path).toLocalFile());
}

void PlacesList::componentComplete()
{
    connect(this, &PlacesList::groupsChanged, this, &PlacesList::setList);
    this->setList();
}

const FMH::MODEL_LIST &PlacesList::items() const
{
    return this->list;
}

FMH::MODEL_LIST PlacesList::getGroup(const KFilePlacesModel &model, const FMStatic::PATHTYPE_KEY &type)
{
    FMH::MODEL_LIST res;

    if (type == FMStatic::PATHTYPE_KEY::QUICK_PATH) {
        res << FMH::MODEL {{FMH::MODEL_KEY::PATH, FMStatic::PATHTYPE_URI[FMStatic::PATHTYPE_KEY::TAGS_PATH] + "fav"}, {FMH::MODEL_KEY::ICON, "love"}, {FMH::MODEL_KEY::LABEL, "Favorite"}, {FMH::MODEL_KEY::TYPE, "Quick"}};

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
    res << FMH::MODEL {{FMH::MODEL_KEY::PATH, "recentdocuments:///"}, {FMH::MODEL_KEY::ICON, "view-media-recent"}, {FMH::MODEL_KEY::LABEL, "Recent"}, {FMH::MODEL_KEY::TYPE, "Quick"}};
#endif

    res << FMH::MODEL {{FMH::MODEL_KEY::PATH, "tags:///"}, {FMH::MODEL_KEY::ICON, "tag"}, {FMH::MODEL_KEY::LABEL, "Tags"}, {FMH::MODEL_KEY::TYPE, "Quick"}};

    return res;
}

if (type == FMStatic::PATHTYPE_KEY::PLACES_PATH) {
    res << FMStatic::getDefaultPaths();
}

#if defined Q_OS_ANDROID || defined Q_OS_WIN32 || defined Q_OS_MACOS || defined Q_OS_IOS
Q_UNUSED(model)
switch (type) {
    case (FMStatic::PATHTYPE_KEY::PLACES_PATH):
        res << FMStatic::packItems(UTIL::loadSettings("BOOKMARKS", "PREFERENCES", {}, true).toStringList(), FMStatic::PATHTYPE_LABEL[FMStatic::PATHTYPE_KEY::BOOKMARKS_PATH]);
        break;
    case (FMStatic::PATHTYPE_KEY::DRIVES_PATH):
        res = FMStatic::getDevices();
        break;
    default:
        break;
}
#else
const auto group = model.groupIndexes(static_cast<KFilePlacesModel::GroupType>(type));
res << std::accumulate(group.constBegin(), group.constEnd(), FMH::MODEL_LIST(), [&model, &type, this](FMH::MODEL_LIST &list, const QModelIndex &index) -> FMH::MODEL_LIST {
    const QUrl url = model.url(index);
    if (type == FMStatic::PATHTYPE_KEY::PLACES_PATH && FMStatic::defaultPaths.contains(url.toString()))
        return list;

    if (type == FMStatic::PATHTYPE_KEY::PLACES_PATH && url.isLocalFile() && !FMH::fileExists(url))
        return list;

    auto data = FMH::MODEL {{FMH::MODEL_KEY::PATH, url.toString()},
    {FMH::MODEL_KEY::URL, url.toString()},
    {FMH::MODEL_KEY::ICON, model.icon(index).name()},
    {FMH::MODEL_KEY::LABEL, model.text(index)},
    {FMH::MODEL_KEY::NAME, model.text(index)},
                       {FMH::MODEL_KEY::TYPE, type == FMStatic::PATHTYPE_KEY::PLACES_PATH ? FMStatic::PATHTYPE_LABEL[FMStatic::PATHTYPE_KEY::BOOKMARKS_PATH] : FMStatic::PATHTYPE_LABEL[type]}};

    if(model.isDevice(index))
    {
        const auto udi =  model.deviceForIndex(index).udi();
        qDebug() << "DEVICE" << udi;

        data.insert(FMH::MODEL_KEY::UDI, udi);
        m_devices.insert(udi, index);
    }

    list << data;
    return list;
});
#endif

return res;
}

void PlacesList::setList()
{
    if (this->groups.isEmpty())
        return;

    qDebug() << "Setting PlacesList model" << groups;
    emit this->preListChanged();

    this->list.clear();

    for (const auto &group : qAsConst(this->groups)) {
        switch (group) {
            case FMStatic::PATHTYPE_KEY::PLACES_PATH:
                this->list << getGroup(*this->model, FMStatic::PATHTYPE_KEY::PLACES_PATH);
            break;

            case FMStatic::PATHTYPE_KEY::QUICK_PATH:
                this->list << getGroup(*this->model, FMStatic::PATHTYPE_KEY::QUICK_PATH);
            break;

            case FMStatic::PATHTYPE_KEY::DRIVES_PATH:
                this->list << getGroup(*this->model, FMStatic::PATHTYPE_KEY::DRIVES_PATH);
            break;

            case FMStatic::PATHTYPE_KEY::REMOTE_PATH:
                this->list << getGroup(*this->model, FMStatic::PATHTYPE_KEY::REMOTE_PATH);
            break;

            case FMStatic::PATHTYPE_KEY::REMOVABLE_PATH:
                this->list << getGroup(*this->model, FMStatic::PATHTYPE_KEY::REMOVABLE_PATH);
            break;

            case FMStatic::PATHTYPE_KEY::TAGS_PATH:            
            this->list << Tagging::getInstance()->getTags();
            break;
                        
#ifdef COMPONENT_ACCOUNTS
            case FMStatic::PATHTYPE_KEY::CLOUD_PATH:
            this->list << MauiAccounts::instance()->getCloudAccounts();
            break;
#endif
        }
    }

    this->setCount();
    emit this->postListChanged();
}

void PlacesList::setCount()
{
    this->watcher->removePaths(this->watcher->directories());
    for (auto &data : this->list) {
        const auto path = data[FMH::MODEL_KEY::URL];
        if (FMStatic::isDir(path)) {
            data.insert(FMH::MODEL_KEY::COUNT, "0");
            QDir dir(QUrl(path).toLocalFile());
            const auto count = dir.count();
            this->count.insert(path, count);
            this->watchPath(path);
        }
    }
}

QList<int> PlacesList::getGroups() const
{
    return this->groups;
}

void PlacesList::setGroups(const QList<int> &value)
{
    if (this->groups == value)
        return;

    this->groups = value;
    emit this->groupsChanged();
}

QVariantMap PlacesList::get(const int &index) const
{
    if (index >= this->list.size() || index < 0)
        return QVariantMap();

    const auto model = this->list.at(index);
    return FMH::toMap(model);
}

void PlacesList::clearBadgeCount(const int &index)
{
    this->list[index][FMH::MODEL_KEY::COUNT] = "0";
    emit this->updateModel(index, {FMH::MODEL_KEY::COUNT});
}

void PlacesList::removePlace(const int &index)
{
    if (index >= this->list.size() || index < 0)
        return;

#if defined Q_OS_ANDROID || defined Q_OS_WIN32 || defined Q_OS_MACOS || defined Q_OS_IOS
    auto bookmarks = UTIL::loadSettings("BOOKMARKS", "PREFERENCES", {}, true).toStringList();
    bookmarks.removeOne(this->list.at(index)[FMH::MODEL_KEY::PATH]);
    UTIL::saveSettings("BOOKMARKS", bookmarks, "PREFERENCES", true);
#else
    emit this->preItemRemoved(index);
    this->model->removePlace(this->model->closestItem(this->list.at(index)[FMH::MODEL_KEY::PATH]));
    this->list.removeAt(index);
    emit this->postItemRemoved();
#endif
}

bool PlacesList::contains(const QUrl &path)
{
    return this->exists(FMH::MODEL_KEY::PATH, path.toString());
}

bool PlacesList::isDevice(const int &index)
{
    if (index >= this->list.size() || index < 0)
        return false;

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
    const auto item = this->list.at(index);
    return m_devices.contains(item[FMH::MODEL_KEY::UDI]);
#endif

    return false;
}

bool PlacesList::setupNeeded(const int &index)
{
    if (index >= this->list.size() || index < 0)
        return false;

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
    const auto item = this->list.at(index);
    if(m_devices.contains(item[FMH::MODEL_KEY::UDI]))
    {
        return this->model->setupNeeded(m_devices.value(item[FMH::MODEL_KEY::UDI]));
    }
#endif

    return false;
}

void PlacesList::requestEject(const int &index)
{
    if (index >= this->list.size() || index < 0)
        return;

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
    const auto item = this->list.at(index);
    if(m_devices.contains(item[FMH::MODEL_KEY::UDI]))
    {
       this->model->requestEject(m_devices.value(item[FMH::MODEL_KEY::UDI]));
    }
#endif
}

void PlacesList::requestSetup(const int &index)
{
    if (index >= this->list.size() || index < 0)
        return;

#if defined Q_OS_LINUX && !defined Q_OS_ANDROID
    const auto item = this->list.at(index);
    if(m_devices.contains(item[FMH::MODEL_KEY::UDI]))
    {
       this->model->requestSetup(m_devices.value(item[FMH::MODEL_KEY::UDI]));
    }
#endif
}

void PlacesList::addBookmark(const QUrl& url)
{
    #if defined Q_OS_ANDROID || defined Q_OS_WIN32 || defined Q_OS_MACOS || defined Q_OS_IOS
    // do android stuff until cmake works with android
    if (FMStatic::isDefaultPath(url.toString()))
        return;
    
        auto bookmarks = UTIL::loadSettings("BOOKMARKS", "PREFERENCES", {}, true).toStringList();
        bookmarks << url.toString();
        UTIL::saveSettings("BOOKMARKS", bookmarks, "PREFERENCES", true);
    #else
    KFilePlacesModel model;
    model.addPlace(QDir(url.toLocalFile()).dirName(), url, FMStatic::getIconName(url));
    #endif
}


