/*
 *
 *  * Copyright (C) 2023, KylinSoft Co., Ltd.
 *  *
 *  * This program is free software: you can redistribute it and/or modify
 *  * it under the terms of the GNU General Public License as published by
 *  * the Free Software Foundation, either version 3 of the License, or
 *  * (at your option) any later version.
 *  *
 *  * This program is distributed in the hope that it will be useful,
 *  * but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  * GNU General Public License for more details.
 *  *
 *  * You should have received a copy of the GNU General Public License
 *  * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *  *
 *  * Authors:  Zhang <zhangyuanyuan1@kylinos.cn>
 *
 */

#ifndef THUMBNAILVIEW_H
#define THUMBNAILVIEW_H

#include <QObject>
#include <QQuickView>
#include <QWindow>
#include <QGSettings>

#include "windowThumbnailModel.h"

enum PanelPositionType {
    Bottom = 0, //!< The bottom side of the screen.
    Top,    //!< The top side of the screen.
    Left,   //!< The left side of the screen.
    Right   //!< The right side of the screen.
};

class ThumbnailView : public QQuickView
{
    Q_OBJECT
    Q_PROPERTY(bool viewVisible READ viewVisible WRITE setViewVisible NOTIFY viewVisibleChanged)
    Q_PROPERTY(bool menuVisible READ menuVisible WRITE setMenuVisible NOTIFY menuVisibleChanged)
    Q_PROPERTY(QList<QVariant> viewModel READ viewModel WRITE setViewModel NOTIFY viewModelChanged)
    Q_PROPERTY(bool isShowByList READ isShowByList WRITE setShowByList NOTIFY showByListChanged)
    Q_PROPERTY(bool isShowHorizontalView READ isShowHorizontalView WRITE setShowHorizontalView NOTIFY showHorizontalViewChanged)
    Q_PROPERTY(int radius READ radius WRITE setRadius NOTIFY radiusChanged)

public:
    explicit ThumbnailView(QWindow *parent = nullptr);
    ~ThumbnailView() = default;

    bool viewVisible() const;
    void setViewVisible(const bool &viewVisible);
    bool menuVisible() const;
    void setMenuVisible(const bool &menuVisible);

    QList<QVariant> viewModel() const;
    bool isShowByList();
    void setShowByList(bool state);
    bool isShowHorizontalView();
    void setShowHorizontalView(bool state);
    int radius();
    void setRadius(int radius);
    void setViewPosition(PanelPositionType position, int panelSize, int x, int y);
    void updataWindowRegion();

    Q_INVOKABLE QVector<QSize> calculateWindowsSize(const QList<QVariant> &model);
    Q_INVOKABLE void updateViewMode();
    Q_INVOKABLE QSize windowSize(const int &index);
    Q_INVOKABLE void updateMprisWindowSize(const int index, int width, int height);
    Q_INVOKABLE void sendSigToKwin(const QVariant &winId);
    Q_INVOKABLE void sendCloseSigToKWin();
    /**
     * @brief 右键菜单
     * @param winId
     */
    Q_INVOKABLE void openMenu(const QVariant &winId);

protected:
    bool event(QEvent *event);

public Q_SLOTS:
    void setViewModel(const QList<QVariant> &model);

private:
    bool m_viewShow = false;
    bool m_menuVisible = false;
    QList<QVariant> m_viewModel;
    bool m_isShowList = false;
    bool m_isHorizontal = true;
    int m_windowRadius = 12;
    int m_totalLenth;
    int m_panelSize = 24;
    QPoint m_viewPoint;
    QVector<QSize> m_winSizeVector;
    QRegion m_windowRegion;

    QVector<QSize> m_mprisWinSize;
    bool m_mprisFlag = false;

    std::unique_ptr<QGSettings> m_styleGsettings;

Q_SIGNALS:
    void viewVisibleChanged(bool viewVisible);
    void menuVisibleChanged(bool m_menuVisible);
    void viewModelChanged(QList<QVariant> viewModel);
    void showByListChanged(bool state);
    void showHorizontalViewChanged(bool state);
    void radiusChanged(int radius);
};

#endif // THUMBNAILVIEW_H
