/*
 * Cray XT/XE system topology definitions
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include <stdint.h>
#include <stdbool.h>

/* These constants are valid for XT and XE */
#define XT_CHASSIS_PER_CABINET	3
#define XT_BLADES_PER_CHASSIS	8
#define XT_BLADES_PER_CABINET	(XT_BLADES_PER_CHASSIS * XT_CHASSIS_PER_CABINET)
#define XT_NODES_PER_BLADE	4
#define XE_NODES_PER_CABINET	(XT_BLADES_PER_CABINET * XT_NODES_PER_BLADE)

/** Cabling variants */
enum cabling_type {
	CABLING_CLASS0,
	CABLING_CLASS1,
	CABLING_CLASS2,
	CABLING_CLASS3,
	CABLING_UNKNOWN
};

/** Torus extension */
enum torus_type {
	TORUS_2D,
	TORUS_3D
};

/** Interconnect variants */
enum asic_type {
	ASIC_XT_SEASTAR,
	ASIC_XE_GEMINI
};

/**
 * struct torus_info - type and corner coordinates of the XT torus
 * @cabling:	cabling class (0..3)
 * @dimension:	dimension (2D or 3D)
 * @network:	torus interconnect (SeaStar or Gemini)
 * @rows:	the number of rows (equals max. row index + 1)
 * @cabs:	the number of cabinets per row (max. cabinet index + 1)
 * @chassis:	the number of chassis (only used for CABLING_CLASS0)
 * @x_max:	maximum X extension (x_max == 0 means 2D Torus)
 * @y_max:	maximum Y extension
 * @z_max:	maximum Z extension
 */
struct torus_info {
	enum cabling_type	cabling;
	enum torus_type 	dimension;
	enum asic_type		network;
	uint8_t			rows;
	uint8_t			cabs;
	uint8_t			chassis;
	uint8_t			x_max;
	uint8_t			y_max;
	uint8_t			z_max;
};

/** Return true if @ti describes a Gemini (XE) system. */
static inline bool is_gemini(const struct torus_info *ti)
{
	return ti->network == ASIC_XE_GEMINI;
}

/**
 * torus_from_placement  -  Determine torus cabling configuration
 * @max_cab:	 highest cabinet index of the system (starting at 0)
 * @max_row:	 highest row index of the system (starting at 0)
 */
static inline enum cabling_type torus_from_placement(uint8_t max_cab,
						     uint8_t max_row)
{
	if (max_row == 0) {
		/*
		 * Class 0 topology:
		 * - single row of 1..3 cabinets, carrying 1..9 chassis/cages
		 * - Torus is N x 4 x 8
		 * - where N = number of chassis
		 */
		if (max_cab < 3)
			return CABLING_CLASS0;
		/*
		 * Class 1 topology:
		 * - single row of 4..16 cabinets
		 * - Torus is N x 12 x 8
		 * - where N = number of cabinets in the row
		 */
		if (max_cab < 16)
			return CABLING_CLASS1;

	} else if (max_row == 1) {
		/*
		 * Class 2 topology:
		 * - 2 rows of equal length
		 * - 16..48 cabinets (i.e., 2*8 .. 2*24 cabinets)
		 * - Torus is N x 12 x 16
		 *   (Y = 12 = 4 SeaStars/blade x 3 cages/cabinet)
		 * - where N = number of cabinets in a row (i.e., 8 <= N <= 24)
		 */
		if (7 <= max_cab && max_cab < 24)
			return CABLING_CLASS2;

	} else if (max_row < 9 && 11 <= max_cab && max_cab < 32) {
		/*
		 * Class 3 topology:
		 * - 3..10 rows of 12..32 cabinets, i.e. 48..320 cabinets total
		 *   o  48 cabinets is  4 rows of 12 cabinets per row
 		 *   o 320 cabinets is 10 rows of 32 cabinets per row
		 * - Torus is N x (4 * R) x 24
		 * - with R >= 3 rows of equal length
 		 * - where N = number of cabinets in a row
		 * - FIXME: when there is an odd number of rows,
		 *   o system has a torus in (X,Z) dimensions and
		 *   o mesh in the Y dimension.
		 */
		return CABLING_CLASS3;
	}
	return CABLING_UNKNOWN;
}

/**
 * struct torus_coord - Node position within the torus
 */
struct torus_coord {
	int x;
	int y;
	int z;
};

/**
 * struct nodecoord - Break down nodename into coordinate information
 * @cab:	cabinet number within row (counted left-to-right)
 * @row:	row number
 * @cage:	chassis within cabinet (counted bottom-to-top); 0..2
 * @slot:	blade within chassis (counted left-to-right);   0..7
 * @node:	node address within slot/blade
 * This struct is used both for SeaStar (XT) and Gemini (XE) systems.
 * Both types contain up to 4 processing nodes. Gemini service/compute blades
 * contain 4 nodes, addressed via Gemini 0/1. SeaStar compute blades have 4
 * nodes n0..n3; SeaStar service blades have two nodes n0/n3.
 */
struct nodecoord {
	uint8_t	cab;
	uint8_t	row;
	uint8_t	cage;
	uint8_t	slot;
	uint8_t node;
};

extern bool is_valid_nodecoord(const struct nodecoord *nc,
			       const struct torus_info *ti);
extern struct torus_coord nid2seastar_toruscoord(const struct nodecoord *nc,
					         const struct torus_info *ti);
extern struct torus_coord nid2gemini_toruscoord(const struct nodecoord *nc,
						const struct torus_info *ti);
extern struct torus_coord nid2toruscoord(const struct nodecoord *nc,
					 const struct torus_info *ti);

extern struct nodecoord *nid2node(const uint32_t nid, struct torus_info *ti);
extern bool string2nodecoord(const char *str, struct nodecoord *nc);
extern const char *nodecoord2string(const struct nodecoord *nc);
extern const char *nodecoord2gemini_string(const struct nodecoord *nc);
extern const char *node_string2gemini_string(const char *nodestr);
