/*
 * Mimic apstat -a
 */
#include "common.h"
#include <time.h>
#include <pwd.h>

/* HELPERS */
extern void print_cmd_detail(const uint8_t *buf_start, appInfo_t *ai_info);

/* Convert @timePlaced into hours/minutes string */
static const char *placement_age(time_t timePlaced)
{
	static char age_str[16];
	uint32_t mins = (time(NULL) - timePlaced) / 60;

	snprintf(age_str, sizeof(age_str),
		 "%dh%02dm", mins/60, mins % 60);
	return age_str;
}

/**
 * print_hex_placement_list  -  print all placement entries, apstat-style
 * @buf_start: start of appinfo buffer area
 * @ai_info:   current appinfo entry
 */
static void print_placement_entries(const uint8_t *buf_start, appInfo_t *ai_info)
{
	placeList_t *entry = (placeList_t *)(buf_start + ai_info->places);
	int j;

	for (j = 0; j < ai_info->numPlaces; j++, entry++) {
#if CLE <= 2
		printf("    PE %d, cmd %u, nid %u, CPU %#x\n", j, entry->cmdIx,
			entry->nid, (unsigned)entry->procMask);
#else
		printf("    PE %d, cmd %u, nid %u, CPU 0x%04x", j, entry->cmdIx,
			entry->nid, (unsigned)entry->procMask);
		printf(", map %#x\n", entry->nodeMap);
#endif
	}
}

/**
 * print_appinfo  -  print single appinfo entry
 * @ai_buf:   start of appinfo buffer
 * @ai_info:  entry to print
 * @inv:      recent ALPS Inventory
 * @num:      number of @ai_info
 * @verbose:  verbosity level
 */
static void print_appinfo(const uint8_t *ai_buf, appInfo_t *ai_info,
			  struct basil_inventory *inv, int num, uint8_t verbose)
{
	cmdDetail_t *cmd = (cmdDetail_t *)(ai_buf + ai_info->cmdDetail);
	const struct basil_rsvn *rsvn = basil_rsvn_by_id(inv, ai_info->resId);

	if (rsvn == NULL)
		fatal("can not find reservation entry in INVENTORY");

	if (verbose == 0) {
		if (ai_info->timePlaced)
			/*     ResId  ApId From      Arch  PEs N d Memory State */
			printf("A%5u %8llu batch:%-10s %s  %4d - - %6u conf,claim\n",
				ai_info->resId,
				(unsigned long long)ai_info->apid,
				rsvn->batch_id,
				nam_arch[rsvn->app_head->cmd_head->arch],
				ai_info->numPlaces,	/* number of PEs requested */
				rsvn->app_head->cmd_head->memory /* per-PE memory  */
			      );
		else
			/*    ResId  ApId From      Arch  PEs N d Memory State */
			printf("%6u %8llu batch:%-10s %s  %4d %d %d %6u %sconf%s\n",
				ai_info->resId,
				(unsigned long long)ai_info->apid,
				rsvn->batch_id,
				nam_arch[rsvn->app_head->cmd_head->arch],
				ai_info->numPlaces,	/* number of PEs requested */
				cmd->fixedPerNode,	/* PEs per node (nppn)	   */
				cmd->depth,		/* number of CPUs per PE   */
				cmd->memory,		/* per-PE memory size	   */
				ai_info->flags & 0x40 ? "NID list," : "",
				num_placed_applications(ai_buf, ai_info->resId) ? ",claim" : ""
				//is_any_node_claimed(ai_buf, ai_info) ? ",claim" : ""
			      );
	} else if (ai_info->timePlaced) {
		struct nodespec *ns;
		/*
		 * apid:    ALPS application ID
		 * pagg:    process aggregate job ID (shell SID or SGI container ID)
		 * account: user account number (often UID)
		 * time:    CPU time limit for the application
		 */
		printf("Ap[%d]: apid %llu, pagg %llu, resId %u, user %s,\n"
		       "       gid %u, account %llu, time %d, normal\n", num,
			(unsigned long long)ai_info->apid,
			(unsigned long long)ai_info->pagg,
			ai_info->resId, rsvn->user_name,
			ai_info->gid,
			(unsigned long long)ai_info->account,
			ai_info->timeLim);

		printf("  Batch System ID = %s\n", rsvn->batch_id);
		printf("  Created at %s", ctime(&rsvn->timestamp));
#if CLE > 2
		if (ai_info->aprunNid)
			printf("  Originator: aprun on NID %d, pid %d\n",
				ai_info->aprunNid, ai_info->aprunPid);
#endif
		printf("  Number of commands %d, control network fanout %d\n",
			ai_info->numCmds, ai_info->fanout);

#if CLE > 2
		printf("  Network: pTag %u, cookie %#x, NTTgran/entries %u/%u, hugePageSz %uM\n",
			ai_info->pTag, ai_info->cookie, ai_info->nttGran, ai_info->numNtt,
			ai_info->hugeTLBPS >> 10);
#endif
		print_cmd_detail(ai_buf, ai_info);
		ns = appinfo_nidlist(ai_buf, ai_info);
#if 1||CLE <= 2
		printf("  Placement list entries: %d\n", ai_info->numPlaces);
#else
		printf("  Placement list entries: %d\n", ns_count_nodes(ns));
#endif
		if (verbose > 2)
			print_placement_entries(ai_buf, ai_info);
		else if (verbose > 1)
			printf("  Placement list: %s\n", ns_to_string(ns));
		free_nodespec(ns);
	}
}

/**
 * apstat_a  -  print placed applications
 * @ai_buf:   start of appinfo buffer area
 */
void apstat_a(const uint8_t *ai_buf, struct basil_inventory *inv,
	      uint64_t apid, unsigned verbosity)
{
	appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	size_t offset;
	int i;

	if (ai_hdr->apNum == 0) {
		printf("No placed applications are present\n");
		return;
	} else {
		printf("Total placed applications: %u\n",
			num_placed_applications(ai_buf, 0));
	}

	printf("Placed  Apid ResId     User   PEs Nodes    Age   State Command\n");
	for (i = 0, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t   *ai_info = (appInfo_t *)(ai_buf + offset);
		cmdDetail_t *cmd = (cmdDetail_t *)(ai_buf + ai_info->cmdDetail);
		struct passwd *pwd = getpwuid(ai_info->uid);

		if (ai_info->timePlaced)
			printf("%12llu %5u %8s %5d %5d %7s  %-5s %-.13s\n",
				(unsigned long long)ai_info->apid,
				ai_info->resId,
				pwd ? pwd->pw_name : "(null)",
				ai_info->numPlaces, cmd->nodeCnt,
				placement_age(ai_info->timePlaced),
				"run",	/* NB: may also be "chkpt" or "wait" */
				cmd->cmd);

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}

	if (verbosity == 0)
		return;
	else if (apid)
		printf("\nApplication detail for apid %llu\n",
			(unsigned long long)apid);
	else
		printf("\nApplication detail\n");

	for (i = 0, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t   *ai_info = (appInfo_t *)(ai_buf + offset);

		if (!apid || ai_info->apid == apid)
			print_appinfo(ai_buf, ai_info, inv, i, verbosity);

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
}
