#![recursion_limit = "1024"]
#![cfg_attr(feature = "cargo-clippy", allow(let_and_return))]

extern crate chrono;
#[macro_use]
extern crate lazy_static;
extern crate regex;

use chrono::{DateTime, FixedOffset, ParseResult};
use std::borrow::Cow;
use regex::Regex;

/// Tries to fix common ways date generators misshandle rfc822/rfc2822.
///
/// For more check the source code, Its ~70 lines of code.
pub fn sanitize_rfc822_like_date<S: Into<String>>(s: S) -> String {
    let s = s.into();
    let s = pad_zeros(s);
    let s = remove_weekday(s);
    let s = replace_month(s);
    let s = replace_leading_zeros(s);
    s
}

/// Pad HH:MM:SS with exta zeros if needed.
fn pad_zeros(s: String) -> String {
    lazy_static! {
        /// If it matchers a pattern of 2:2:2, return.
        static ref OK_RGX: Regex = Regex::new(r"(\d{2}):(\d{2}):(\d{2})").unwrap();

        /// hours, minutes, seconds = cap[1], cap[2], cap[3]
        static ref RE_RGX: Regex = Regex::new(r"(\d{1,2}):(\d{1,2}):(\d{1,2})").unwrap();
    }

    if OK_RGX.is_match(&s) {
        return s;
    }

    if let Some(cap) = RE_RGX.captures(&s) {
        let mut tm = String::with_capacity(2 + 1 + 2 + 1 + 2 + 1);
        cap.iter().skip(1).filter_map(|m| m).for_each(|mtch| {
            let m_str = mtch.as_str();
            if m_str.len() == 1 {
                tm.push('0');
            }
            tm.push_str(m_str);
            tm.push(':');
        });
        tm.pop(); // Pop leftover last separator (at no penalty, since we only allocate once
                  // either way)

        return s.replace(&cap[0], &tm);
    }

    s
}

/// Weekday name is not required for rfc2822
fn remove_weekday(s: String) -> String {
    static WEEKDAYS: &[&str] = &[
        "Mon,",
        "Tue,",
        "Wed,",
        "Thu,",
        "Fri,",
        "Sat,",
        "Sun,",
        "Monday,",
        "Tuesday,",
        "Wednesday,",
        "Thursday,",
        "Friday,",
        "Saturday,",
        "Sunday,",
    ];

    WEEKDAYS
        .iter()
        .find(|&w| s.starts_with(w))
        .map(|w| s[w.len()..].trim().to_string())
        .unwrap_or(s)
}

/// Replace long month names with 3 letter Abr as specified in RFC2822.
fn replace_month(s: String) -> String {
    static MONTHS: &[(&str, &str)] = &[
        ("January", "Jan"),
        ("February", "Feb"),
        ("March", "Mar"),
        ("April ", "Apr"),
        ("May", "May"),
        ("June", "Jun"),
        ("July", "Jul"),
        ("August", "Aug"),
        ("September", "Sep"),
        ("October", "Oct"),
        ("November", "Nov"),
        ("December", "Dec"),
    ];

    MONTHS
        .iter()
        .find(|&&(k, _)| s.contains(k))
        .map(|&(k, v)| s.replace(k, v))
        .unwrap_or(s)
}

/// Convert -0000 to +0000.
/// See [#102](https://github.com/chronotope/chrono/issues/102)
fn replace_leading_zeros(s: String) -> String {
    if s.ends_with("-0000") {
        format!("{}+0000", &s[..s.len() - 5])
    } else {
        s
    }
}


/// Calls `DateTime::parse_from_rfc2822()`. If it succedes returns,
/// Else it calls `sanitize_rfc822_like_date` and retries.
///
/// Basic usage:
///
/// ```rust
/// # extern crate chrono; extern crate rfc822_sanitizer;
/// # use chrono::DateTime;
/// # use rfc822_sanitizer::parse_from_rfc2822_with_fallback;
///
/// # fn main() {
/// let bad_input = parse_from_rfc2822_with_fallback("Thu, 05 Aug 2016 06:00:00 -0400");
/// let correct_result = DateTime::parse_from_rfc2822("Fri, 05 Aug 2016 06:00:00 -0400");
/// assert_eq!(bad_input, correct_result);
/// # }
/// ```
pub fn parse_from_rfc2822_with_fallback<'s, S: Into<Cow<'s, str>>>(
    s: S,
) -> ParseResult<DateTime<FixedOffset>> {
    let s = s.into();
    let date = DateTime::parse_from_rfc2822(&s);
    match date {
        Ok(_) => date,
        Err(err) => {
            let san = sanitize_rfc822_like_date(s);
            if let Ok(dt) = DateTime::parse_from_rfc2822(&san) {
                return Ok(dt);
            }
            Err(err)
        }
    }
}


#[cfg(test)]
mod tests {
    use super::*;
    use chrono::DateTime;

    #[test]
    fn test_invalid_dates() {
        // left is raw date extracted from rss feeds.
        // right is corresponding valid rfc2822
        let dates = vec![
            ("Thu, 6 July 2017 15:30:00 PDT", "6 Jul 2017 15:30:00 PDT"),
            ("Mon, 10 July 2017 16:00:00 PDT", "10 Jul 2017 16:00:00 PDT"),
            ("Mon, 17 July 2017 17:00:00 PDT", "17 Jul 2017 17:00:00 PDT"),
            ("Mon, 24 July 2017 16:00:00 PDT", "24 Jul 2017 16:00:00 PDT"),
            ("Mon, 31 July 2017 16:00:00 PDT", "31 Jul 2017 16:00:00 PDT"),
            ("Thu, 30 Aug 2017 1:30:00 PDT", "30 Aug 2017 01:30:00 PDT"),
            (
                "Wed, 20 Sep 2017 10:00:00 -0000",
                "20 Sep 2017 10:00:00 +0000",
            ),
            (
                "Wed, 13 Sep 2017 10:00:00 -0000",
                "13 Sep 2017 10:00:00 +0000",
            ),
            (
                "Wed, 09 Aug 2017 10:00:00 -0000",
                "09 Aug 2017 10:00:00 +0000",
            ),
            (
                "Wed, 02 Aug 2017 10:00:00 -0000",
                "02 Aug 2017 10:00:00 +0000",
            ),
            (
                "Wed, 26 Jul 2017 10:00:00 -0000",
                "26 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 19 Jul 2017 10:00:00 -0000",
                "19 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 12 Jul 2017 10:00:00 -0000",
                "12 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 28 Jun 2017 10:00:00 -0000",
                "28 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 21 Jun 2017 10:00:00 -0000",
                "21 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 14 Jun 2017 10:00:00 -0000",
                "14 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 07 Jun 2017 10:00:00 -0000",
                "07 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 31 May 2017 10:00:00 -0000",
                "31 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 24 May 2017 10:00:00 -0000",
                "24 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 17 May 2017 10:00:00 -0000",
                "17 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 10 May 2017 10:00:00 -0000",
                "10 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 03 May 2017 10:00:00 -0000",
                "03 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 19 Apr 2017 10:00:00 -0000",
                "19 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 12 Apr 2017 10:00:00 -0000",
                "12 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 05 Apr 2017 10:00:00 -0000",
                "05 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 29 Mar 2017 10:00:00 -0000",
                "29 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 22 Mar 2017 10:00:00 -0000",
                "22 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 15 Mar 2017 10:00:00 -0000",
                "15 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 08 Mar 2017 11:00:00 -0000",
                "08 Mar 2017 11:00:00 +0000",
            ),
            (
                "Wed, 01 Mar 2017 11:00:00 -0000",
                "01 Mar 2017 11:00:00 +0000",
            ),
            (
                "Wed, 22 Feb 2017 11:00:00 -0000",
                "22 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 15 Feb 2017 11:00:00 -0000",
                "15 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 08 Feb 2017 11:00:00 -0000",
                "08 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 01 Feb 2017 11:00:00 -0000",
                "01 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 25 Jan 2017 11:00:00 -0000",
                "25 Jan 2017 11:00:00 +0000",
            ),
            (
                "Fri, 13 Jan 2017 18:38:00 -0000",
                "13 Jan 2017 18:38:00 +0000",
            ),
            (
                "Wed, 20 Sep 2017 03:30:00 -0000",
                "20 Sep 2017 03:30:00 +0000",
            ),
            (
                "Wed, 13 Sep 2017 03:15:00 -0000",
                "13 Sep 2017 03:15:00 +0000",
            ),
            (
                "Wed, 06 Sep 2017 03:15:00 -0000",
                "06 Sep 2017 03:15:00 +0000",
            ),
            (
                "Wed, 30 Aug 2017 03:15:00 -0000",
                "30 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 23 Aug 2017 03:15:00 -0000",
                "23 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 16 Aug 2017 03:15:00 -0000",
                "16 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 09 Aug 2017 03:15:00 -0000",
                "09 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 02 Aug 2017 03:00:00 -0000",
                "02 Aug 2017 03:00:00 +0000",
            ),
            (
                "Tue, 11 Jul 2017 17:14:45 -0000",
                "11 Jul 2017 17:14:45 +0000",
            ),
            (
                "Thu, 03 August 2017 06:00:00 -0400",
                "03 Aug 2017 06:00:00 -0400",
            ),
            (
                "Thu, 27 July 2017 06:00:00 -0400",
                "27 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 20 July 2017 06:00:00 -0400",
                "20 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 13 July 2017 06:00:00 -0400",
                "13 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 06 July 2017 06:00:00 -0400",
                "06 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 28 June 2017 06:00:00 -0400",
                "28 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 17 Jul 2013 06:00:03 -0400",
                "17 Jul 2013 06:00:03 -0400",
            ),
            (
                "Thu, 02 Apr 2014 06:00:03 -0400",
                "02 Apr 2014 06:00:03 -0400",
            ),
            (
                "Wed, 14 Jan 2016 06:00:03 -0400",
                "14 Jan 2016 06:00:03 -0400",
            ),
            (
                "Thu, 22 June 2017 06:00:00 -0400",
                "22 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 June 2017 06:00:00 -0400",
                "15 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 7 June 2017 06:00:00 -0400",
                "7 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 1 June 2017 06:00:00 -0400",
                "1 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 23 Dec 2015 06:00:03 -0400",
                "23 Dec 2015 06:00:03 -0400",
            ),
            (
                "Thu, 14 Feb 2014 06:00:03 -0400",
                "14 Feb 2014 06:00:03 -0400",
            ),
            (
                "Thu, 04 Dec 2013 06:00:03 -0400",
                "04 Dec 2013 06:00:03 -0400",
            ),
            (
                "Thu, 20 Dec 2016 06:00:00 -0400",
                "20 Dec 2016 06:00:00 -0400",
            ),
            (
                "Thu, 23 Nov 2016 06:00:00 -0400",
                "23 Nov 2016 06:00:00 -0400",
            ),
            (
                "Thu, 05 Aug 2016 06:00:00 -0400",
                "05 Aug 2016 06:00:00 -0400",
            ),
            (
                "Fri, 09 Jun 2016 12:00:00 -0400",
                "09 Jun 2016 12:00:00 -0400",
            ),
            (
                "Thu, 10 May 2017 06:00:00 -0400",
                "10 May 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 Feb 2017 06:00:00 -0400",
                "22 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 Feb 2017 06:00:00 -0400",
                "15 Feb 2017 06:00:00 -0400",
            ),
        ];

        dates.iter().for_each(|&(bad, good)| {
            assert_eq!(
                parse_from_rfc2822_with_fallback(bad),
                DateTime::parse_from_rfc2822(good)
            )
        });
    }


    #[test]
    fn test_sanitize_rfc822_like_date() {
        // left is raw date extracted from rss feeds.
        // right is corresponding valid rfc2822
        let dates = vec![
            ("Thu, 6 July 2017 15:30:00 PDT", "6 Jul 2017 15:30:00 PDT"),
            ("Mon, 10 July 2017 16:00:00 PDT", "10 Jul 2017 16:00:00 PDT"),
            ("Mon, 17 July 2017 17:00:00 PDT", "17 Jul 2017 17:00:00 PDT"),
            ("Mon, 24 July 2017 16:00:00 PDT", "24 Jul 2017 16:00:00 PDT"),
            ("Mon, 31 July 2017 16:00:00 PDT", "31 Jul 2017 16:00:00 PDT"),
            ("Thu, 30 Aug 2017 1:30:00 PDT", "30 Aug 2017 01:30:00 PDT"),
            (
                "Wed, 20 Sep 2017 10:00:00 -0000",
                "20 Sep 2017 10:00:00 +0000",
            ),
            (
                "Wed, 13 Sep 2017 10:00:00 -0000",
                "13 Sep 2017 10:00:00 +0000",
            ),
            (
                "Wed, 09 Aug 2017 10:00:00 -0000",
                "09 Aug 2017 10:00:00 +0000",
            ),
            (
                "Wed, 02 Aug 2017 10:00:00 -0000",
                "02 Aug 2017 10:00:00 +0000",
            ),
            (
                "Wed, 26 Jul 2017 10:00:00 -0000",
                "26 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 19 Jul 2017 10:00:00 -0000",
                "19 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 12 Jul 2017 10:00:00 -0000",
                "12 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 28 Jun 2017 10:00:00 -0000",
                "28 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 21 Jun 2017 10:00:00 -0000",
                "21 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 14 Jun 2017 10:00:00 -0000",
                "14 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 07 Jun 2017 10:00:00 -0000",
                "07 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 31 May 2017 10:00:00 -0000",
                "31 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 24 May 2017 10:00:00 -0000",
                "24 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 17 May 2017 10:00:00 -0000",
                "17 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 10 May 2017 10:00:00 -0000",
                "10 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 03 May 2017 10:00:00 -0000",
                "03 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 19 Apr 2017 10:00:00 -0000",
                "19 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 12 Apr 2017 10:00:00 -0000",
                "12 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 05 Apr 2017 10:00:00 -0000",
                "05 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 29 Mar 2017 10:00:00 -0000",
                "29 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 22 Mar 2017 10:00:00 -0000",
                "22 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 15 Mar 2017 10:00:00 -0000",
                "15 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 08 Mar 2017 11:00:00 -0000",
                "08 Mar 2017 11:00:00 +0000",
            ),
            (
                "Wed, 01 Mar 2017 11:00:00 -0000",
                "01 Mar 2017 11:00:00 +0000",
            ),
            (
                "Wed, 22 Feb 2017 11:00:00 -0000",
                "22 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 15 Feb 2017 11:00:00 -0000",
                "15 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 08 Feb 2017 11:00:00 -0000",
                "08 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 01 Feb 2017 11:00:00 -0000",
                "01 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 25 Jan 2017 11:00:00 -0000",
                "25 Jan 2017 11:00:00 +0000",
            ),
            (
                "Fri, 13 Jan 2017 18:38:00 -0000",
                "13 Jan 2017 18:38:00 +0000",
            ),
            (
                "Wed, 20 Sep 2017 03:30:00 -0000",
                "20 Sep 2017 03:30:00 +0000",
            ),
            (
                "Wed, 13 Sep 2017 03:15:00 -0000",
                "13 Sep 2017 03:15:00 +0000",
            ),
            (
                "Wed, 06 Sep 2017 03:15:00 -0000",
                "06 Sep 2017 03:15:00 +0000",
            ),
            (
                "Wed, 30 Aug 2017 03:15:00 -0000",
                "30 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 23 Aug 2017 03:15:00 -0000",
                "23 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 16 Aug 2017 03:15:00 -0000",
                "16 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 09 Aug 2017 03:15:00 -0000",
                "09 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 02 Aug 2017 03:00:00 -0000",
                "02 Aug 2017 03:00:00 +0000",
            ),
            (
                "Tue, 11 Jul 2017 17:14:45 -0000",
                "11 Jul 2017 17:14:45 +0000",
            ),
            (
                "Thu, 03 August 2017 06:00:00 -0400",
                "03 Aug 2017 06:00:00 -0400",
            ),
            (
                "Thu, 27 July 2017 06:00:00 -0400",
                "27 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 20 July 2017 06:00:00 -0400",
                "20 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 13 July 2017 06:00:00 -0400",
                "13 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 06 July 2017 06:00:00 -0400",
                "06 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 28 June 2017 06:00:00 -0400",
                "28 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 17 Jul 2013 06:00:03 -0400",
                "17 Jul 2013 06:00:03 -0400",
            ),
            (
                "Thu, 02 Apr 2014 06:00:03 -0400",
                "02 Apr 2014 06:00:03 -0400",
            ),
            (
                "Wed, 14 Jan 2016 06:00:03 -0400",
                "14 Jan 2016 06:00:03 -0400",
            ),
            (
                "Thu, 22 June 2017 06:00:00 -0400",
                "22 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 June 2017 06:00:00 -0400",
                "15 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 7 June 2017 06:00:00 -0400",
                "7 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 1 June 2017 06:00:00 -0400",
                "1 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 23 Dec 2015 06:00:03 -0400",
                "23 Dec 2015 06:00:03 -0400",
            ),
            (
                "Thu, 14 Feb 2014 06:00:03 -0400",
                "14 Feb 2014 06:00:03 -0400",
            ),
            (
                "Thu, 04 Dec 2013 06:00:03 -0400",
                "04 Dec 2013 06:00:03 -0400",
            ),
            (
                "Thu, 20 Dec 2016 06:00:00 -0400",
                "20 Dec 2016 06:00:00 -0400",
            ),
            (
                "Thu, 23 Nov 2016 06:00:00 -0400",
                "23 Nov 2016 06:00:00 -0400",
            ),
            (
                "Thu, 05 Aug 2016 06:00:00 -0400",
                "05 Aug 2016 06:00:00 -0400",
            ),
            (
                "Fri, 09 Jun 2016 12:00:00 -0400",
                "09 Jun 2016 12:00:00 -0400",
            ),
            (
                "Thu, 10 May 2017 06:00:00 -0400",
                "10 May 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 Feb 2017 06:00:00 -0400",
                "22 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 Feb 2017 06:00:00 -0400",
                "15 Feb 2017 06:00:00 -0400",
            ),
        ];

        dates.iter().for_each(|&(bad, good)| {
            assert_eq!(sanitize_rfc822_like_date(bad.to_string()), good)
        });
    }

    #[test]
    fn test_remove_weekday() {
        let foo = vec![
            ("Thu, 6 July 2017 15:30:00 PDT", "6 July 2017 15:30:00 PDT"),
            (
                "Mon, 10 July 2017 16:00:00 PDT",
                "10 July 2017 16:00:00 PDT",
            ),
            (
                "Mon, 17 July 2017 17:00:00 PDT",
                "17 July 2017 17:00:00 PDT",
            ),
            (
                "Mon, 24 July 2017 16:00:00 PDT",
                "24 July 2017 16:00:00 PDT",
            ),
            (
                "Mon, 31 July 2017 16:00:00 PDT",
                "31 July 2017 16:00:00 PDT",
            ),
            ("Thu, 30 Aug 2017 1:30:00 PDT", "30 Aug 2017 1:30:00 PDT"),
            (
                "Wed, 20 Sep 2017 10:00:00 -0000",
                "20 Sep 2017 10:00:00 -0000",
            ),
            (
                "Wed, 13 Sep 2017 10:00:00 -0000",
                "13 Sep 2017 10:00:00 -0000",
            ),
            (
                "Wed, 09 Aug 2017 10:00:00 -0000",
                "09 Aug 2017 10:00:00 -0000",
            ),
            (
                "Wed, 02 Aug 2017 10:00:00 -0000",
                "02 Aug 2017 10:00:00 -0000",
            ),
            (
                "Wed, 26 Jul 2017 10:00:00 -0000",
                "26 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 19 Jul 2017 10:00:00 -0000",
                "19 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 12 Jul 2017 10:00:00 -0000",
                "12 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 28 Jun 2017 10:00:00 -0000",
                "28 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 21 Jun 2017 10:00:00 -0000",
                "21 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 14 Jun 2017 10:00:00 -0000",
                "14 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 07 Jun 2017 10:00:00 -0000",
                "07 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 31 May 2017 10:00:00 -0000",
                "31 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 24 May 2017 10:00:00 -0000",
                "24 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 17 May 2017 10:00:00 -0000",
                "17 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 10 May 2017 10:00:00 -0000",
                "10 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 03 May 2017 10:00:00 -0000",
                "03 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 19 Apr 2017 10:00:00 -0000",
                "19 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 12 Apr 2017 10:00:00 -0000",
                "12 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 05 Apr 2017 10:00:00 -0000",
                "05 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 29 Mar 2017 10:00:00 -0000",
                "29 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 22 Mar 2017 10:00:00 -0000",
                "22 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 15 Mar 2017 10:00:00 -0000",
                "15 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 08 Mar 2017 11:00:00 -0000",
                "08 Mar 2017 11:00:00 -0000",
            ),
            (
                "Wed, 01 Mar 2017 11:00:00 -0000",
                "01 Mar 2017 11:00:00 -0000",
            ),
            (
                "Wed, 22 Feb 2017 11:00:00 -0000",
                "22 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 15 Feb 2017 11:00:00 -0000",
                "15 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 08 Feb 2017 11:00:00 -0000",
                "08 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 01 Feb 2017 11:00:00 -0000",
                "01 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 25 Jan 2017 11:00:00 -0000",
                "25 Jan 2017 11:00:00 -0000",
            ),
            (
                "Fri, 13 Jan 2017 18:38:00 -0000",
                "13 Jan 2017 18:38:00 -0000",
            ),
            (
                "Wed, 20 Sep 2017 03:30:00 -0000",
                "20 Sep 2017 03:30:00 -0000",
            ),
            (
                "Wed, 13 Sep 2017 03:15:00 -0000",
                "13 Sep 2017 03:15:00 -0000",
            ),
            (
                "Wed, 06 Sep 2017 03:15:00 -0000",
                "06 Sep 2017 03:15:00 -0000",
            ),
            (
                "Wed, 30 Aug 2017 03:15:00 -0000",
                "30 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 23 Aug 2017 03:15:00 -0000",
                "23 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 16 Aug 2017 03:15:00 -0000",
                "16 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 09 Aug 2017 03:15:00 -0000",
                "09 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 02 Aug 2017 03:00:00 -0000",
                "02 Aug 2017 03:00:00 -0000",
            ),
            (
                "Tue, 11 Jul 2017 17:14:45 -0000",
                "11 Jul 2017 17:14:45 -0000",
            ),
            (
                "Thu, 03 August 2017 06:00:00 -0400",
                "03 August 2017 06:00:00 -0400",
            ),
            (
                "Thu, 27 July 2017 06:00:00 -0400",
                "27 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 20 July 2017 06:00:00 -0400",
                "20 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 13 July 2017 06:00:00 -0400",
                "13 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 06 July 2017 06:00:00 -0400",
                "06 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 28 June 2017 06:00:00 -0400",
                "28 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 17 Jul 2013 06:00:03 -0400",
                "17 Jul 2013 06:00:03 -0400",
            ),
            (
                "Thu, 02 Apr 2014 06:00:03 -0400",
                "02 Apr 2014 06:00:03 -0400",
            ),
            (
                "Wed, 14 Jan 2016 06:00:03 -0400",
                "14 Jan 2016 06:00:03 -0400",
            ),
            (
                "Thu, 22 June 2017 06:00:00 -0400",
                "22 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 June 2017 06:00:00 -0400",
                "15 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 7 June 2017 06:00:00 -0400",
                "7 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 1 June 2017 06:00:00 -0400",
                "1 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 23 Dec 2015 06:00:03 -0400",
                "23 Dec 2015 06:00:03 -0400",
            ),
            (
                "Thu, 14 Feb 2014 06:00:03 -0400",
                "14 Feb 2014 06:00:03 -0400",
            ),
            (
                "Thu, 04 Dec 2013 06:00:03 -0400",
                "04 Dec 2013 06:00:03 -0400",
            ),
            (
                "Thu, 20 Dec 2016 06:00:00 -0400",
                "20 Dec 2016 06:00:00 -0400",
            ),
            (
                "Thu, 23 Nov 2016 06:00:00 -0400",
                "23 Nov 2016 06:00:00 -0400",
            ),
            (
                "Thu, 05 Aug 2016 06:00:00 -0400",
                "05 Aug 2016 06:00:00 -0400",
            ),
            (
                "Fri, 09 Jun 2016 12:00:00 -0400",
                "09 Jun 2016 12:00:00 -0400",
            ),
            (
                "Thu, 10 May 2017 06:00:00 -0400",
                "10 May 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 Feb 2017 06:00:00 -0400",
                "22 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 Feb 2017 06:00:00 -0400",
                "15 Feb 2017 06:00:00 -0400",
            ),
        ];

        foo.iter().for_each(|&(bad, good)| {
            assert_eq!(remove_weekday(bad.to_string()), good)
        });
    }

    #[test]
    fn test_pad_zeros() {
        // Would be nice If we had more test cases,
        // If you stumble(d) upon any online please consider opening a Pullrequest.
        let foo = vec![
            (
                "Thu, 30 Aug 2017 1:30:00 PDT",
                "Thu, 30 Aug 2017 01:30:00 PDT",
            ),
        ];

        foo.iter()
            .for_each(|&(bad, good)| assert_eq!(pad_zeros(bad.to_string()), good));
    }

    #[test]
    fn test_replace_month() {
        let foo = vec![
            (
                "Thu, 6 July 2017 15:30:00 PDT",
                "Thu, 6 Jul 2017 15:30:00 PDT",
            ),
            (
                "Thu, 6 July 2017 15:30:00 PDT",
                "Thu, 6 Jul 2017 15:30:00 PDT",
            ),
            (
                "Mon, 10 July 2017 16:00:00 PDT",
                "Mon, 10 Jul 2017 16:00:00 PDT",
            ),
            (
                "Mon, 10 July 2017 16:00:00 PDT",
                "Mon, 10 Jul 2017 16:00:00 PDT",
            ),
            (
                "Mon, 17 July 2017 17:00:00 PDT",
                "Mon, 17 Jul 2017 17:00:00 PDT",
            ),
            (
                "Mon, 17 July 2017 17:00:00 PDT",
                "Mon, 17 Jul 2017 17:00:00 PDT",
            ),
            (
                "Mon, 24 July 2017 16:00:00 PDT",
                "Mon, 24 Jul 2017 16:00:00 PDT",
            ),
            (
                "Mon, 24 July 2017 16:00:00 PDT",
                "Mon, 24 Jul 2017 16:00:00 PDT",
            ),
            (
                "Mon, 31 July 2017 16:00:00 PDT",
                "Mon, 31 Jul 2017 16:00:00 PDT",
            ),
            (
                "Mon, 31 July 2017 16:00:00 PDT",
                "Mon, 31 Jul 2017 16:00:00 PDT",
            ),
            (
                "Thu, 30 Aug 2017 1:30:00 PDT",
                "Thu, 30 Aug 2017 1:30:00 PDT",
            ),
            (
                "Thu, 30 Aug 2017 1:30:00 PDT",
                "Thu, 30 Aug 2017 1:30:00 PDT",
            ),
            (
                "Wed, 20 Sep 2017 10:00:00 -0000",
                "Wed, 20 Sep 2017 10:00:00 -0000",
            ),
            (
                "Wed, 20 Sep 2017 10:00:00 -0000",
                "Wed, 20 Sep 2017 10:00:00 -0000",
            ),
            (
                "Wed, 13 Sep 2017 10:00:00 -0000",
                "Wed, 13 Sep 2017 10:00:00 -0000",
            ),
            (
                "Wed, 13 Sep 2017 10:00:00 -0000",
                "Wed, 13 Sep 2017 10:00:00 -0000",
            ),
            (
                "Wed, 09 Aug 2017 10:00:00 -0000",
                "Wed, 09 Aug 2017 10:00:00 -0000",
            ),
            (
                "Wed, 09 Aug 2017 10:00:00 -0000",
                "Wed, 09 Aug 2017 10:00:00 -0000",
            ),
            (
                "Wed, 02 Aug 2017 10:00:00 -0000",
                "Wed, 02 Aug 2017 10:00:00 -0000",
            ),
            (
                "Wed, 02 Aug 2017 10:00:00 -0000",
                "Wed, 02 Aug 2017 10:00:00 -0000",
            ),
            (
                "Wed, 26 Jul 2017 10:00:00 -0000",
                "Wed, 26 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 26 Jul 2017 10:00:00 -0000",
                "Wed, 26 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 19 Jul 2017 10:00:00 -0000",
                "Wed, 19 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 19 Jul 2017 10:00:00 -0000",
                "Wed, 19 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 12 Jul 2017 10:00:00 -0000",
                "Wed, 12 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 12 Jul 2017 10:00:00 -0000",
                "Wed, 12 Jul 2017 10:00:00 -0000",
            ),
            (
                "Wed, 28 Jun 2017 10:00:00 -0000",
                "Wed, 28 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 28 Jun 2017 10:00:00 -0000",
                "Wed, 28 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 21 Jun 2017 10:00:00 -0000",
                "Wed, 21 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 21 Jun 2017 10:00:00 -0000",
                "Wed, 21 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 14 Jun 2017 10:00:00 -0000",
                "Wed, 14 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 14 Jun 2017 10:00:00 -0000",
                "Wed, 14 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 07 Jun 2017 10:00:00 -0000",
                "Wed, 07 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 07 Jun 2017 10:00:00 -0000",
                "Wed, 07 Jun 2017 10:00:00 -0000",
            ),
            (
                "Wed, 31 May 2017 10:00:00 -0000",
                "Wed, 31 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 31 May 2017 10:00:00 -0000",
                "Wed, 31 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 24 May 2017 10:00:00 -0000",
                "Wed, 24 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 24 May 2017 10:00:00 -0000",
                "Wed, 24 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 17 May 2017 10:00:00 -0000",
                "Wed, 17 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 17 May 2017 10:00:00 -0000",
                "Wed, 17 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 10 May 2017 10:00:00 -0000",
                "Wed, 10 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 10 May 2017 10:00:00 -0000",
                "Wed, 10 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 03 May 2017 10:00:00 -0000",
                "Wed, 03 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 03 May 2017 10:00:00 -0000",
                "Wed, 03 May 2017 10:00:00 -0000",
            ),
            (
                "Wed, 19 Apr 2017 10:00:00 -0000",
                "Wed, 19 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 19 Apr 2017 10:00:00 -0000",
                "Wed, 19 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 12 Apr 2017 10:00:00 -0000",
                "Wed, 12 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 12 Apr 2017 10:00:00 -0000",
                "Wed, 12 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 05 Apr 2017 10:00:00 -0000",
                "Wed, 05 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 05 Apr 2017 10:00:00 -0000",
                "Wed, 05 Apr 2017 10:00:00 -0000",
            ),
            (
                "Wed, 29 Mar 2017 10:00:00 -0000",
                "Wed, 29 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 29 Mar 2017 10:00:00 -0000",
                "Wed, 29 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 22 Mar 2017 10:00:00 -0000",
                "Wed, 22 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 22 Mar 2017 10:00:00 -0000",
                "Wed, 22 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 15 Mar 2017 10:00:00 -0000",
                "Wed, 15 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 15 Mar 2017 10:00:00 -0000",
                "Wed, 15 Mar 2017 10:00:00 -0000",
            ),
            (
                "Wed, 08 Mar 2017 11:00:00 -0000",
                "Wed, 08 Mar 2017 11:00:00 -0000",
            ),
            (
                "Wed, 08 Mar 2017 11:00:00 -0000",
                "Wed, 08 Mar 2017 11:00:00 -0000",
            ),
            (
                "Wed, 01 Mar 2017 11:00:00 -0000",
                "Wed, 01 Mar 2017 11:00:00 -0000",
            ),
            (
                "Wed, 01 Mar 2017 11:00:00 -0000",
                "Wed, 01 Mar 2017 11:00:00 -0000",
            ),
            (
                "Wed, 22 Feb 2017 11:00:00 -0000",
                "Wed, 22 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 22 Feb 2017 11:00:00 -0000",
                "Wed, 22 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 15 Feb 2017 11:00:00 -0000",
                "Wed, 15 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 15 Feb 2017 11:00:00 -0000",
                "Wed, 15 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 08 Feb 2017 11:00:00 -0000",
                "Wed, 08 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 08 Feb 2017 11:00:00 -0000",
                "Wed, 08 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 01 Feb 2017 11:00:00 -0000",
                "Wed, 01 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 01 Feb 2017 11:00:00 -0000",
                "Wed, 01 Feb 2017 11:00:00 -0000",
            ),
            (
                "Wed, 25 Jan 2017 11:00:00 -0000",
                "Wed, 25 Jan 2017 11:00:00 -0000",
            ),
            (
                "Wed, 25 Jan 2017 11:00:00 -0000",
                "Wed, 25 Jan 2017 11:00:00 -0000",
            ),
            (
                "Fri, 13 Jan 2017 18:38:00 -0000",
                "Fri, 13 Jan 2017 18:38:00 -0000",
            ),
            (
                "Fri, 13 Jan 2017 18:38:00 -0000",
                "Fri, 13 Jan 2017 18:38:00 -0000",
            ),
            (
                "Wed, 20 Sep 2017 03:30:00 -0000",
                "Wed, 20 Sep 2017 03:30:00 -0000",
            ),
            (
                "Wed, 20 Sep 2017 03:30:00 -0000",
                "Wed, 20 Sep 2017 03:30:00 -0000",
            ),
            (
                "Wed, 13 Sep 2017 03:15:00 -0000",
                "Wed, 13 Sep 2017 03:15:00 -0000",
            ),
            (
                "Wed, 13 Sep 2017 03:15:00 -0000",
                "Wed, 13 Sep 2017 03:15:00 -0000",
            ),
            (
                "Wed, 06 Sep 2017 03:15:00 -0000",
                "Wed, 06 Sep 2017 03:15:00 -0000",
            ),
            (
                "Wed, 06 Sep 2017 03:15:00 -0000",
                "Wed, 06 Sep 2017 03:15:00 -0000",
            ),
            (
                "Wed, 30 Aug 2017 03:15:00 -0000",
                "Wed, 30 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 30 Aug 2017 03:15:00 -0000",
                "Wed, 30 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 23 Aug 2017 03:15:00 -0000",
                "Wed, 23 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 23 Aug 2017 03:15:00 -0000",
                "Wed, 23 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 16 Aug 2017 03:15:00 -0000",
                "Wed, 16 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 16 Aug 2017 03:15:00 -0000",
                "Wed, 16 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 09 Aug 2017 03:15:00 -0000",
                "Wed, 09 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 09 Aug 2017 03:15:00 -0000",
                "Wed, 09 Aug 2017 03:15:00 -0000",
            ),
            (
                "Wed, 02 Aug 2017 03:00:00 -0000",
                "Wed, 02 Aug 2017 03:00:00 -0000",
            ),
            (
                "Wed, 02 Aug 2017 03:00:00 -0000",
                "Wed, 02 Aug 2017 03:00:00 -0000",
            ),
            (
                "Tue, 11 Jul 2017 17:14:45 -0000",
                "Tue, 11 Jul 2017 17:14:45 -0000",
            ),
            (
                "Tue, 11 Jul 2017 17:14:45 -0000",
                "Tue, 11 Jul 2017 17:14:45 -0000",
            ),
            (
                "Thu, 03 August 2017 06:00:00 -0400",
                "Thu, 03 Aug 2017 06:00:00 -0400",
            ),
            (
                "Thu, 03 August 2017 06:00:00 -0400",
                "Thu, 03 Aug 2017 06:00:00 -0400",
            ),
            (
                "Thu, 27 July 2017 06:00:00 -0400",
                "Thu, 27 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 27 July 2017 06:00:00 -0400",
                "Thu, 27 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 20 July 2017 06:00:00 -0400",
                "Thu, 20 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 20 July 2017 06:00:00 -0400",
                "Thu, 20 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 13 July 2017 06:00:00 -0400",
                "Thu, 13 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 13 July 2017 06:00:00 -0400",
                "Thu, 13 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 06 July 2017 06:00:00 -0400",
                "Thu, 06 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 06 July 2017 06:00:00 -0400",
                "Thu, 06 Jul 2017 06:00:00 -0400",
            ),
            (
                "Thu, 28 June 2017 06:00:00 -0400",
                "Thu, 28 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 28 June 2017 06:00:00 -0400",
                "Thu, 28 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 17 Jul 2013 06:00:03 -0400",
                "Thu, 17 Jul 2013 06:00:03 -0400",
            ),
            (
                "Thu, 17 Jul 2013 06:00:03 -0400",
                "Thu, 17 Jul 2013 06:00:03 -0400",
            ),
            (
                "Thu, 02 Apr 2014 06:00:03 -0400",
                "Thu, 02 Apr 2014 06:00:03 -0400",
            ),
            (
                "Thu, 02 Apr 2014 06:00:03 -0400",
                "Thu, 02 Apr 2014 06:00:03 -0400",
            ),
            (
                "Wed, 14 Jan 2016 06:00:03 -0400",
                "Wed, 14 Jan 2016 06:00:03 -0400",
            ),
            (
                "Wed, 14 Jan 2016 06:00:03 -0400",
                "Wed, 14 Jan 2016 06:00:03 -0400",
            ),
            (
                "Thu, 22 June 2017 06:00:00 -0400",
                "Thu, 22 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 June 2017 06:00:00 -0400",
                "Thu, 22 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 June 2017 06:00:00 -0400",
                "Thu, 15 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 June 2017 06:00:00 -0400",
                "Thu, 15 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 7 June 2017 06:00:00 -0400",
                "Thu, 7 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 7 June 2017 06:00:00 -0400",
                "Thu, 7 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 1 June 2017 06:00:00 -0400",
                "Thu, 1 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 1 June 2017 06:00:00 -0400",
                "Thu, 1 Jun 2017 06:00:00 -0400",
            ),
            (
                "Thu, 23 Dec 2015 06:00:03 -0400",
                "Thu, 23 Dec 2015 06:00:03 -0400",
            ),
            (
                "Thu, 23 Dec 2015 06:00:03 -0400",
                "Thu, 23 Dec 2015 06:00:03 -0400",
            ),
            (
                "Thu, 14 Feb 2014 06:00:03 -0400",
                "Thu, 14 Feb 2014 06:00:03 -0400",
            ),
            (
                "Thu, 14 Feb 2014 06:00:03 -0400",
                "Thu, 14 Feb 2014 06:00:03 -0400",
            ),
            (
                "Thu, 04 Dec 2013 06:00:03 -0400",
                "Thu, 04 Dec 2013 06:00:03 -0400",
            ),
            (
                "Thu, 04 Dec 2013 06:00:03 -0400",
                "Thu, 04 Dec 2013 06:00:03 -0400",
            ),
            (
                "Thu, 20 Dec 2016 06:00:00 -0400",
                "Thu, 20 Dec 2016 06:00:00 -0400",
            ),
            (
                "Thu, 20 Dec 2016 06:00:00 -0400",
                "Thu, 20 Dec 2016 06:00:00 -0400",
            ),
            (
                "Thu, 23 Nov 2016 06:00:00 -0400",
                "Thu, 23 Nov 2016 06:00:00 -0400",
            ),
            (
                "Thu, 23 Nov 2016 06:00:00 -0400",
                "Thu, 23 Nov 2016 06:00:00 -0400",
            ),
            (
                "Thu, 05 Aug 2016 06:00:00 -0400",
                "Thu, 05 Aug 2016 06:00:00 -0400",
            ),
            (
                "Thu, 05 Aug 2016 06:00:00 -0400",
                "Thu, 05 Aug 2016 06:00:00 -0400",
            ),
            (
                "Fri, 09 Jun 2016 12:00:00 -0400",
                "Fri, 09 Jun 2016 12:00:00 -0400",
            ),
            (
                "Fri, 09 Jun 2016 12:00:00 -0400",
                "Fri, 09 Jun 2016 12:00:00 -0400",
            ),
            (
                "Thu, 10 May 2017 06:00:00 -0400",
                "Thu, 10 May 2017 06:00:00 -0400",
            ),
            (
                "Thu, 10 May 2017 06:00:00 -0400",
                "Thu, 10 May 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 Feb 2017 06:00:00 -0400",
                "Thu, 22 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 Feb 2017 06:00:00 -0400",
                "Thu, 22 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 Feb 2017 06:00:00 -0400",
                "Thu, 15 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 Feb 2017 06:00:00 -0400",
                "Thu, 15 Feb 2017 06:00:00 -0400",
            ),
        ];

        foo.iter().for_each(|&(bad, good)| {
            assert_eq!(replace_month(bad.to_string()), good)
        });
    }

    #[test]
    fn test_replace_leading_zeroes() {
        let foo = vec![
            (
                "Thu, 6 July 2017 15:30:00 PDT",
                "Thu, 6 July 2017 15:30:00 PDT",
            ),
            (
                "Mon, 10 July 2017 16:00:00 PDT",
                "Mon, 10 July 2017 16:00:00 PDT",
            ),
            (
                "Mon, 17 July 2017 17:00:00 PDT",
                "Mon, 17 July 2017 17:00:00 PDT",
            ),
            (
                "Mon, 24 July 2017 16:00:00 PDT",
                "Mon, 24 July 2017 16:00:00 PDT",
            ),
            (
                "Mon, 31 July 2017 16:00:00 PDT",
                "Mon, 31 July 2017 16:00:00 PDT",
            ),
            (
                "Thu, 30 Aug 2017 1:30:00 PDT",
                "Thu, 30 Aug 2017 1:30:00 PDT",
            ),
            (
                "Wed, 20 Sep 2017 10:00:00 -0000",
                "Wed, 20 Sep 2017 10:00:00 +0000",
            ),
            (
                "Wed, 13 Sep 2017 10:00:00 -0000",
                "Wed, 13 Sep 2017 10:00:00 +0000",
            ),
            (
                "Wed, 09 Aug 2017 10:00:00 -0000",
                "Wed, 09 Aug 2017 10:00:00 +0000",
            ),
            (
                "Wed, 02 Aug 2017 10:00:00 -0000",
                "Wed, 02 Aug 2017 10:00:00 +0000",
            ),
            (
                "Wed, 26 Jul 2017 10:00:00 -0000",
                "Wed, 26 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 19 Jul 2017 10:00:00 -0000",
                "Wed, 19 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 12 Jul 2017 10:00:00 -0000",
                "Wed, 12 Jul 2017 10:00:00 +0000",
            ),
            (
                "Wed, 28 Jun 2017 10:00:00 -0000",
                "Wed, 28 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 21 Jun 2017 10:00:00 -0000",
                "Wed, 21 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 14 Jun 2017 10:00:00 -0000",
                "Wed, 14 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 07 Jun 2017 10:00:00 -0000",
                "Wed, 07 Jun 2017 10:00:00 +0000",
            ),
            (
                "Wed, 31 May 2017 10:00:00 -0000",
                "Wed, 31 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 24 May 2017 10:00:00 -0000",
                "Wed, 24 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 17 May 2017 10:00:00 -0000",
                "Wed, 17 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 10 May 2017 10:00:00 -0000",
                "Wed, 10 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 03 May 2017 10:00:00 -0000",
                "Wed, 03 May 2017 10:00:00 +0000",
            ),
            (
                "Wed, 19 Apr 2017 10:00:00 -0000",
                "Wed, 19 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 12 Apr 2017 10:00:00 -0000",
                "Wed, 12 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 05 Apr 2017 10:00:00 -0000",
                "Wed, 05 Apr 2017 10:00:00 +0000",
            ),
            (
                "Wed, 29 Mar 2017 10:00:00 -0000",
                "Wed, 29 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 22 Mar 2017 10:00:00 -0000",
                "Wed, 22 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 15 Mar 2017 10:00:00 -0000",
                "Wed, 15 Mar 2017 10:00:00 +0000",
            ),
            (
                "Wed, 08 Mar 2017 11:00:00 -0000",
                "Wed, 08 Mar 2017 11:00:00 +0000",
            ),
            (
                "Wed, 01 Mar 2017 11:00:00 -0000",
                "Wed, 01 Mar 2017 11:00:00 +0000",
            ),
            (
                "Wed, 22 Feb 2017 11:00:00 -0000",
                "Wed, 22 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 15 Feb 2017 11:00:00 -0000",
                "Wed, 15 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 08 Feb 2017 11:00:00 -0000",
                "Wed, 08 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 01 Feb 2017 11:00:00 -0000",
                "Wed, 01 Feb 2017 11:00:00 +0000",
            ),
            (
                "Wed, 25 Jan 2017 11:00:00 -0000",
                "Wed, 25 Jan 2017 11:00:00 +0000",
            ),
            (
                "Fri, 13 Jan 2017 18:38:00 -0000",
                "Fri, 13 Jan 2017 18:38:00 +0000",
            ),
            (
                "Wed, 20 Sep 2017 03:30:00 -0000",
                "Wed, 20 Sep 2017 03:30:00 +0000",
            ),
            (
                "Wed, 13 Sep 2017 03:15:00 -0000",
                "Wed, 13 Sep 2017 03:15:00 +0000",
            ),
            (
                "Wed, 06 Sep 2017 03:15:00 -0000",
                "Wed, 06 Sep 2017 03:15:00 +0000",
            ),
            (
                "Wed, 30 Aug 2017 03:15:00 -0000",
                "Wed, 30 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 23 Aug 2017 03:15:00 -0000",
                "Wed, 23 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 16 Aug 2017 03:15:00 -0000",
                "Wed, 16 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 09 Aug 2017 03:15:00 -0000",
                "Wed, 09 Aug 2017 03:15:00 +0000",
            ),
            (
                "Wed, 02 Aug 2017 03:00:00 -0000",
                "Wed, 02 Aug 2017 03:00:00 +0000",
            ),
            (
                "Tue, 11 Jul 2017 17:14:45 -0000",
                "Tue, 11 Jul 2017 17:14:45 +0000",
            ),
            (
                "Thu, 03 August 2017 06:00:00 -0400",
                "Thu, 03 August 2017 06:00:00 -0400",
            ),
            (
                "Thu, 27 July 2017 06:00:00 -0400",
                "Thu, 27 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 20 July 2017 06:00:00 -0400",
                "Thu, 20 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 13 July 2017 06:00:00 -0400",
                "Thu, 13 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 06 July 2017 06:00:00 -0400",
                "Thu, 06 July 2017 06:00:00 -0400",
            ),
            (
                "Thu, 28 June 2017 06:00:00 -0400",
                "Thu, 28 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 17 Jul 2013 06:00:03 -0400",
                "Thu, 17 Jul 2013 06:00:03 -0400",
            ),
            (
                "Thu, 02 Apr 2014 06:00:03 -0400",
                "Thu, 02 Apr 2014 06:00:03 -0400",
            ),
            (
                "Wed, 14 Jan 2016 06:00:03 -0400",
                "Wed, 14 Jan 2016 06:00:03 -0400",
            ),
            (
                "Thu, 22 June 2017 06:00:00 -0400",
                "Thu, 22 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 June 2017 06:00:00 -0400",
                "Thu, 15 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 7 June 2017 06:00:00 -0400",
                "Thu, 7 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 1 June 2017 06:00:00 -0400",
                "Thu, 1 June 2017 06:00:00 -0400",
            ),
            (
                "Thu, 23 Dec 2015 06:00:03 -0400",
                "Thu, 23 Dec 2015 06:00:03 -0400",
            ),
            (
                "Thu, 14 Feb 2014 06:00:03 -0400",
                "Thu, 14 Feb 2014 06:00:03 -0400",
            ),
            (
                "Thu, 04 Dec 2013 06:00:03 -0400",
                "Thu, 04 Dec 2013 06:00:03 -0400",
            ),
            (
                "Thu, 20 Dec 2016 06:00:00 -0400",
                "Thu, 20 Dec 2016 06:00:00 -0400",
            ),
            (
                "Thu, 23 Nov 2016 06:00:00 -0400",
                "Thu, 23 Nov 2016 06:00:00 -0400",
            ),
            (
                "Thu, 05 Aug 2016 06:00:00 -0400",
                "Thu, 05 Aug 2016 06:00:00 -0400",
            ),
            (
                "Fri, 09 Jun 2016 12:00:00 -0400",
                "Fri, 09 Jun 2016 12:00:00 -0400",
            ),
            (
                "Thu, 10 May 2017 06:00:00 -0400",
                "Thu, 10 May 2017 06:00:00 -0400",
            ),
            (
                "Thu, 22 Feb 2017 06:00:00 -0400",
                "Thu, 22 Feb 2017 06:00:00 -0400",
            ),
            (
                "Thu, 15 Feb 2017 06:00:00 -0400",
                "Thu, 15 Feb 2017 06:00:00 -0400",
            ),
        ];

        foo.iter().for_each(|&(bad, good)| {
            assert_eq!(replace_leading_zeros(bad.to_string()), good)
        });
    }
}
