/*
    SPDX-FileCopyrightText: 2018-2019 Kai Uwe Broulik <kde@privat.broulik.de>

    SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only OR LicenseRef-KDE-Accepted-GPL
*/

pragma ComponentBehavior: Bound

import QtQuick
import QtQuick.Layouts

import org.kde.plasma.plasmoid
import org.kde.plasma.core as PlasmaCore
import org.kde.ksvg as KSvg
import org.kde.plasma.components as PlasmaComponents3
import org.kde.plasma.extras as PlasmaExtras
import org.kde.kirigami as Kirigami

import org.kde.coreaddons as KCoreAddons

import org.kde.notificationmanager as NotificationManager
import plasma.applet.org.kde.plasma.notifications as Notifications

import "delegates" as Delegates
import "components" as Components

PlasmaExtras.Representation {
    id: fullRepresentationRoot

    // TODO these should be configurable in the future
    readonly property int dndMorningHour: 6
    readonly property int dndEveningHour: 20
    required property PlasmoidItem appletInterface
    required property NotificationManager.Settings notificationSettings
    required property PlasmaCore.Action clearHistoryAction
    required property NotificationManager.Notifications historyModel

    Layout.minimumWidth: Kirigami.Units.gridUnit * 12
    Layout.minimumHeight: Kirigami.Units.gridUnit * 12
    Layout.preferredWidth: Kirigami.Units.gridUnit * 18
    Layout.preferredHeight: Kirigami.Units.gridUnit * 24
    Layout.maximumWidth: Kirigami.Units.gridUnit * 80
    Layout.maximumHeight: Kirigami.Units.gridUnit * 40

    Layout.fillHeight: Plasmoid.formFactor === PlasmaCore.Types.Vertical

    collapseMarginsHint: true

    Keys.onDownPressed: dndCheck.forceActiveFocus(Qt.TabFocusReason);

    Connections {
        target: fullRepresentationRoot.appletInterface
        function onExpandedChanged() {
            if (fullRepresentationRoot.appletInterface.expanded) {
                list.positionViewAtBeginning();
                list.currentIndex = -1;
                for (let i = 0; i < fullRepresentationRoot.historyModel.count; i++)  {
                    let rowIdx = fullRepresentationRoot.historyModel.index(i, 0);
                    fullRepresentationRoot.historyModel.setData(rowIdx, true, NotificationManager.Notifications.ReadRole);
                }
            }
        }
    }

    header: PlasmaExtras.PlasmoidHeading {
        ColumnLayout {
            anchors {
                fill: parent
                leftMargin: Kirigami.Units.smallSpacing
            }
            id: header
            spacing: 0

            RowLayout {
                Layout.fillWidth: true
                spacing: 0

                PlasmaComponents3.Switch {
                    id: dndCheck
                    enabled: NotificationManager.Server.valid
                    text: i18n("Do not disturb")
                    icon.name: "notifications-disabled"
                    checkable: true
                    checked: Notifications.Globals.inhibited

                    Accessible.onPressAction: if (Notifications.Globals.inhibited) {
                        Notifications.Globals.revokeInhibitions();
                    } else {
                        let date = new Date();
                        date.setFullYear(date.getFullYear() + 1);
                        fullRepresentationRoot.notificationSettings.notificationsInhibitedUntil = date;
                        fullRepresentationRoot.notificationSettings.save();
                    }
                    KeyNavigation.down: list
                    KeyNavigation.tab: list

                    // Let the menu open on press
                    onPressed: {
                        if (!Notifications.Globals.inhibited) {
                            dndMenu.date = new Date();
                            // shows ontop of CheckBox to hide the fact that it's unchecked
                            // until you actually select something :)
                            dndMenu.open(0, 0);
                        }
                    }
                    // but disable only on click
                    onClicked: {
                        if (Notifications.Globals.inhibited) {
                            Notifications.Globals.revokeInhibitions();
                        }
                    }


                    PlasmaExtras.ModelContextMenu {
                        id: dndMenu
                        property date date
                        visualParent: dndCheck

                        onClicked: {
                            fullRepresentationRoot.notificationSettings.notificationsInhibitedUntil = model.date;
                            fullRepresentationRoot.notificationSettings.save();
                        }

                        model: {
                            var model = [];

                            // For 1 hour
                            var d = dndMenu.date;
                            d.setHours(d.getHours() + 1);
                            d.setSeconds(0);
                            model.push({date: d, text: i18n("For 1 hour")});

                            d = dndMenu.date;
                            d.setHours(d.getHours() + 4);
                            d.setSeconds(0);
                            model.push({date: d, text: i18n("For 4 hours")});

                            // Until this evening
                            if (dndMenu.date.getHours() < fullRepresentationRoot.dndEveningHour) {
                                d = dndMenu.date;
                                // TODO make the user's preferred time schedule configurable
                                d.setHours(fullRepresentationRoot.dndEveningHour);
                                d.setMinutes(0);
                                d.setSeconds(0);
                                model.push({date: d, text: i18n("Until this evening")});
                            }

                            // Until next morning
                            if (dndMenu.date.getHours() > fullRepresentationRoot.dndMorningHour) {
                                d = dndMenu.date;
                                d.setDate(d.getDate() + 1);
                                d.setHours(fullRepresentationRoot.dndMorningHour);
                                d.setMinutes(0);
                                d.setSeconds(0);
                                model.push({date: d, text: i18n("Until tomorrow morning")});
                            }

                            // Until Monday
                            // show Friday and Saturday, Sunday is "0" but for that you can use "until tomorrow morning"
                            if (dndMenu.date.getDay() >= 5) {
                                d = dndMenu.date;
                                d.setHours(fullRepresentationRoot.dndMorningHour);
                                // wraps around if necessary
                                d.setDate(d.getDate() + (7 - d.getDay() + 1));
                                d.setMinutes(0);
                                d.setSeconds(0);
                                model.push({date: d, text: i18n("Until Monday")});
                            }

                            // Until "turned off"
                            d = dndMenu.date;
                            // Just set it to one year in the future so we don't need yet another "do not disturb enabled" property
                            d.setFullYear(d.getFullYear() + 1);
                            model.push({date: d, text: i18n("Until manually disabled")});

                            return model;
                        }
                    }
                }

                Item {
                    Layout.fillWidth: true
                }

                PlasmaComponents3.ToolButton {
                    visible: !(Plasmoid.containmentDisplayHints & PlasmaCore.Types.ContainmentDrawsPlasmoidHeading)

                    Accessible.name: fullRepresentationRoot.clearHistoryAction.text
                    icon.name: "edit-clear-history"
                    enabled: fullRepresentationRoot.clearHistoryAction.visible
                    onClicked: fullRepresentationRoot.clearHistoryAction.trigger()

                    PlasmaComponents3.ToolTip {
                        text: parent.Accessible.name
                    }
                }
            }

            PlasmaExtras.DescriptiveLabel {
                Layout.leftMargin: dndCheck.mirrored ? 0 : dndCheck.indicator.width + 2 * dndCheck.spacing + Kirigami.Units.iconSizes.smallMedium
                Layout.rightMargin: dndCheck.mirrored ? dndCheck.indicator.width + 2 * dndCheck.spacing + Kirigami.Units.iconSizes.smallMedium : 0
                Layout.fillWidth: true
                wrapMode: Text.WordWrap
                textFormat: Text.PlainText
                text: {
                    if (!Notifications.Globals.inhibited) {
                        return "";
                    }

                    var inhibitedUntil = fullRepresentationRoot.notificationSettings.notificationsInhibitedUntil;
                    var inhibitedUntilTime = inhibitedUntil.getTime();
                    var inhibitedByApp = fullRepresentationRoot.notificationSettings.notificationsInhibitedByApplication;
                    var inhibitedByMirroredScreens = fullRepresentationRoot.notificationSettings.inhibitNotificationsWhenScreensMirrored
                                                        && fullRepresentationRoot.notificationSettings.screensMirrored;
                    var inhibitedByFullscreen = fullRepresentationRoot.notificationSettings.inhibitNotificationsWhenFullscreen
                                                        && fullRepresentationRoot.notificationSettings.fullscreenFocused;
                    var dateNow = Date.now();

                    var sections = [];

                    // Show until time if valid but not if too far int he future
                    if (!isNaN(inhibitedUntilTime) && inhibitedUntilTime - dateNow > 0 &&
                        inhibitedUntilTime - dateNow < 100 * 24 * 60 * 60 * 1000 /* 1 year*/) {
                        const endTime = KCoreAddons.Format.formatRelativeDateTime(inhibitedUntil, Locale.ShortFormat);
                        const lowercaseEndTime = endTime[0] + endTime.slice(1);
                        sections.push(i18nc("Do not disturb until date", "Automatically ends: %1", lowercaseEndTime));
                    }

                    if (inhibitedByApp) {
                        var inhibitionAppNames = fullRepresentationRoot.notificationSettings.notificationInhibitionApplications;
                        var inhibitionAppReasons = fullRepresentationRoot.notificationSettings.notificationInhibitionReasons;

                        for (var i = 0, length = inhibitionAppNames.length; i < length; ++i) {
                            var name = inhibitionAppNames[i];
                            var reason = inhibitionAppReasons[i];

                            if (reason) {
                                sections.push(i18nc("Do not disturb until app has finished (reason)", "While %1 is active (%2)", name, reason));
                            } else {
                                sections.push(i18nc("Do not disturb until app has finished", "While %1 is active", name));
                            }
                        }
                    }

                    if (inhibitedByMirroredScreens) {
                        sections.push(i18nc("Do not disturb because external mirrored screens connected", "Screens are mirrored"))
                    }

                    if (inhibitedByFullscreen) {
                        sections.push(i18nc("Do not disturb because fullscreen app is focused", "Fullscreen app is focused"))
                    }

                    return sections.join(" · ");
                }
                visible: text !== ""
            }
        }
    }

    PlasmaComponents3.ScrollView {
        id: scrollView
        anchors.fill: parent
        background: null
        focus: true

        contentItem: ListView {
            id: list
            width: scrollView.availableWidth
            focus: true
            model: fullRepresentationRoot.appletInterface.expanded ? fullRepresentationRoot.historyModel : null
            currentIndex: -1

            topMargin: Kirigami.Units.largeSpacing
            bottomMargin: Kirigami.Units.largeSpacing
            spacing: Kirigami.Units.smallSpacing

            KeyNavigation.up: dndCheck

            Keys.onDeletePressed: {
                var idx = fullRepresentationRoot.historyModel.index(currentIndex, 0);
                if (fullRepresentationRoot.historyModel.data(idx, NotificationManager.Notifications.ClosableRole)) {
                    fullRepresentationRoot.historyModel.close(idx);
                    // TODO would be nice to stay inside the current group when deleting an item
                }
            }
            Keys.onEnterPressed: event => { Keys.returnPressed(event) }
            Keys.onReturnPressed: {
                // Trigger default action, if any
                var idx = fullRepresentationRoot.historyModel.index(currentIndex, 0);
                if (fullRepresentationRoot.historyModel.data(idx, NotificationManager.Notifications.HasDefaultActionRole)) {
                    fullRepresentationRoot.historyModel.invokeDefaultAction(idx);
                    return;
                }

                // Trigger thumbnail URL if there's one
                var urls = fullRepresentationRoot.historyModel.data(idx, NotificationManager.Notifications.UrlsRole);
                if (urls && urls.length === 1) {
                    Qt.openUrlExternally(urls[0]);
                    return;
                }

                // TODO for finished jobs trigger "Open" or "Open Containing Folder" action
            }
            Keys.onLeftPressed: setGroupExpanded(currentIndex, LayoutMirroring.enabled)
            Keys.onRightPressed: setGroupExpanded(currentIndex, !LayoutMirroring.enabled)

            Keys.onPressed: event => {
                switch (event.key) {
                case Qt.Key_Home:
                    currentIndex = 0;
                    break;
                case Qt.Key_End:
                    currentIndex = count - 1;
                    break;
                }
            }

            function setGroupExpanded(row, expanded) {
                var rowIdx = fullRepresentationRoot.historyModel.index(row, 0);
                var persistentRowIdx = fullRepresentationRoot.historyModel.makePersistentModelIndex(rowIdx);
                var persistentGroupIdx = fullRepresentationRoot.historyModel.makePersistentModelIndex(fullRepresentationRoot.historyModel.groupIndex(rowIdx));

                fullRepresentationRoot.historyModel.setData(rowIdx, expanded, NotificationManager.Notifications.IsGroupExpandedRole);

                // If the current item went away when the group collapsed, scroll to the group heading
                if (!persistentRowIdx || !persistentRowIdx.valid) {
                    if (persistentGroupIdx && persistentGroupIdx.valid) {
                        list.positionViewAtIndex(persistentGroupIdx.row, ListView.Contain);
                        // When closed via keyboard, also set a sane current index
                        if (list.currentIndex > -1) {
                            list.currentIndex = persistentGroupIdx.row;
                        }
                    }
                }
            }

            highlightMoveDuration: 0
            highlightResizeDuration: 0
            // Not using PlasmaExtras.Highlight as this is only for indicating keyboard focus
            highlight: KSvg.FrameSvgItem {
                imagePath: "widgets/listitem"
                prefix: "pressed"
            }

            section {
                property: "desktopEntry"
                criteria: ViewSection.FullString
                delegate: Item {
                    width: list.width
                    // Within a section delegate we don't have other ways to detect whether we are the first section
                    height: y > 0 ? Math.round(Kirigami.Units.smallSpacing * 2) : 0

                    KSvg.SvgItem {
                        anchors {
                            left: parent.left
                            right: parent.right
                            verticalCenter: parent.verticalCenter
                            margins: Kirigami.Units.largeSpacing
                        }
                        imagePath: "widgets/line"
                        elementId: "horizontal-line"
                        visible: parent.y > 0
                    }
                }
            }

            delegate: DraggableDelegate {
                id: delegate
                width: ListView.view.width
                opacity: 0

                required property int index
                required property var model

                required property bool isGroup
                required property bool isInGroup
                required property int type
                required property bool hasDefaultAction
                required property list<string> actionLabels
                required property string configureActionLabel
                required property bool hasReplyAction
                required property string applicationName
                required property string applicationIconName
                required property string originName
                required property date updated
                required property date created
                required property bool configurable
                required property bool dismissable
                required property bool dismissed
                required property bool closable
                required property string summary
                required property string body
                required property var image
                required property string iconName
                required property list<url> urls
                required property string defaultActionLabel
                required property int jobState
                required property int percentage
                required property string jobError
                required property bool suspendable
                required property bool killable
                required property QtObject jobDetails
                required property list<string> actionNames
                required property bool resident
                required property bool isGroupExpanded
                required property int groupChildrenCount
                required property int expandedGroupChildrenCount

                // NOTE: The following animations replace the Transitions in the ListView
                // because they don't work when the items change size during the animation
                // (showing/hiding the show more/show less button) in that case they will
                // animate to a wrong position and stay there
                // see https://bugs.kde.org/show_bug.cgi?id=427894 and QTBUG-110366
                property real oldY: -1
                property int oldListCount: -1
                onYChanged: {
                    if (oldY < 0 || oldListCount === list.count) {
                        oldY = y;
                        return;
                    }
                    traslAnim.from = oldY - y;
                    //traslAnim.running = true;
                    traslAnim.restart()
                    oldY = y;
                    oldListCount = list.count;
                }
                transform: Translate {
                    id: transl
                }
                NumberAnimation {
                    id: traslAnim
                    target: transl
                    properties: "y"
                    to: 0
                    duration: Kirigami.Units.longDuration
                }

                ListView.onAdd: appearAnim.restart();
                Component.onCompleted: {
                    Qt.callLater(() => {
                        if (!appearAnim.running) {
                            opacity = 1;
                        }
                    });
                    oldListCount = list.count;
                }

                SequentialAnimation {
                    id: appearAnim
                    PropertyAction { target: delegate; property: "opacity"; value: 0 }
                    PauseAnimation { duration: Kirigami.Units.longDuration}
                    NumberAnimation {
                        target: delegate
                        property: "opacity"
                        from: 0
                        to: 1
                        duration: Kirigami.Units.longDuration
                    }
                }

                draggable: !delegate.isGroup && delegate.type != NotificationManager.Notifications.JobType

                onDismissRequested: fullRepresentationRoot.historyModel.close(fullRepresentationRoot.historyModel.index(index, 0));

                contentItem: Loader {
                    id: delegateLoader

                    sourceComponent: {
                        if (delegate.isGroup) {
                            return groupDelegate;
                        } else if (delegate.isInGroup) {
                            return notificationGroupedDelegate;
                        } else {
                            return notificationDelegate;
                        }
                    }

                    readonly property Components.ModelInterface modelInterface: Components.ModelInterface {
                        index: delegate.index
                        notificationType: delegate.type

                        hasSomeActions: (delegate.hasDefaultAction || false) || (delegate.actionLabels || []).length > 0 || (delegate.configureActionLabel || "").length > 0 || (delegate.hasReplyAction || false)
                        hasReplyAction: delegate.hasReplyAction || false

                        inGroup: delegate.isInGroup
                        inHistory: true

                        applicationName: delegate.applicationName
                        applicationIconSource: delegate.applicationIconName
                        originName: delegate.originName || ""

                        time: isNaN(delegate.updated) ? delegate.created : delegate.updated

                        // configure button on every single notifications is bit overwhelming
                        configurable: !inGroup && delegate.configurable

                        dismissable: delegate.dismissable
                            && delegate.dismissed
                            // TODO would be nice to be able to undismiss jobs even when they autohide
                            && fullRepresentationRoot.notificationSettings.permanentJobPopups
                        dismissed: delegate.dismissed || false
                        closable: delegate.closable

                        summary: delegate.summary
                        body: delegate.body || ""
                        icon: delegate.image || delegate.iconName

                        urls: delegate.urls || []

                        defaultActionLabel: delegate.defaultActionLabel || i18nc("@action:button", "View")

                        // In the popup the default action is triggered by clicking on the popup
                        // however in the list this is undesirable, so instead show a clickable button
                        // in case you have a non-expired notification in history (do not disturb mode)
                        // unless it has the same label as an action
                        addDefaultAction: (delegate.hasDefaultAction && (delegate.actionLabels || []).indexOf(delegate.defaultActionLabel || i18nc("@action:button", "View")) === -1) ? true : false

                        jobState: delegate.jobState || 0
                        percentage: delegate.percentage || 0
                        jobError: delegate.jobError || 0
                        suspendable: !!delegate.suspendable
                        killable: !!delegate.killable
                        jobDetails: delegate.jobDetails || null

                        configureActionLabel: delegate.configureActionLabel || ""

                        actionNames: delegate.actionNames || []
                        actionLabels: delegate.actionLabels || []

                        onCloseClicked: delegate.close()

                        onDismissClicked: {
                            delegate.model.dismissed = false;
                            fullRepresentationRoot.appletInterface.closePlasmoid();
                        }
                        onConfigureClicked: fullRepresentationRoot.historyModel.configure(fullRepresentationRoot.historyModel.index(index, 0))

                        onActionInvoked: actionName => {
                            // We close any non-resident notification (even those that still may have some actions)
                            // because the assumption is that once the notification has been interacted with, it may
                            // safely lose interaction capabilities (since the user is now likely in the app itself).
                            //
                            // The alternative to this would have the downside that notifications whose apps have been
                            // closed will keep their buttons in the notification history. This way, invoking an action
                            // will make the notification actually disappear (as is common on other operating systems).
                            const behavior = delegate.resident ? NotificationManager.Notifications.None : NotificationManager.Notifications.Close;

                            if (actionName === "default") {
                                fullRepresentationRoot.historyModel.invokeDefaultAction(fullRepresentationRoot.historyModel.index(index, 0), behavior);
                            } else {
                                fullRepresentationRoot.historyModel.invokeAction(fullRepresentationRoot.historyModel.index(index, 0), actionName, behavior);
                            }
                        }
                        onOpenUrl: url => {
                            Qt.openUrlExternally(url);
                            delegateLoader.expire();
                        }
                        onFileActionInvoked: action => {
                            if (action.objectName === "movetotrash" || action.objectName === "deletefile") {
                                delegate.close();
                            } else {
                                delegateLoader.expire();
                            }
                        }
                        onReplied: text => {
                            const behavior = delegate.resident ? NotificationManager.Notifications.None : NotificationManager.Notifications.Close;
                            fullRepresentationRoot.historyModel.reply(fullRepresentationRoot.historyModel.index(index, 0), text, behavior);
                        }

                        onSuspendJobClicked: fullRepresentationRoot.historyModel.suspendJob(fullRepresentationRoot.historyModel.index(index, 0))
                        onResumeJobClicked: fullRepresentationRoot.historyModel.resumeJob(fullRepresentationRoot.historyModel.index(index, 0))
                        onKillJobClicked: fullRepresentationRoot.historyModel.killJob(fullRepresentationRoot.historyModel.index(index, 0))
                    }

                    function expire() {
                        if (delegate.resident) {
                            delegate.model.expired = true;
                        } else {
                            fullRepresentationRoot.historyModel.expire(fullRepresentationRoot.historyModel.index(delegate.index, 0));
                        }
                    }

                    Component {
                        id: groupDelegate
                        Components.NotificationHeader {
                            modelInterface {
                                applicationName: delegate.applicationName
                                applicationIconSource: delegate.applicationIconName
                                originName: delegate.originName || ""

                                configurable: delegate.configurable
                                closable: delegate.closable

                                onCloseClicked: fullRepresentationRoot.historyModel.close(fullRepresentationRoot.historyModel.index(delegate.index, 0));
                                onConfigureClicked: fullRepresentationRoot.historyModel.configure(fullRepresentationRoot.historyModel.index(delegate.index, 0))
                            }
                            closeButtonTooltip: i18n("Close Group")
                        }
                    }
                    Component {
                        id: notificationDelegate
                        Delegates.DelegateHistory {
                            Layout.fillWidth: true
                            modelInterface: delegateLoader.modelInterface
                        }
                    }
                    Component {
                        id: notificationGroupedDelegate
                        ColumnLayout {
                            spacing: Kirigami.Units.smallSpacing

                            Delegates.DelegateHistoryGrouped {
                                Layout.fillWidth: true
                                modelInterface: delegateLoader.modelInterface
                            }

                            PlasmaComponents3.ToolButton {
                                icon.name: delegate.isGroupExpanded ? "arrow-up" : "arrow-down"
                                text: delegate.isGroupExpanded ? i18n("Show Fewer")
                                                            : i18nc("Expand to show n more notifications",
                                                                    "Show %1 More", (delegate.groupChildrenCount - delegate.expandedGroupChildrenCount))
                                visible: (delegate.groupChildrenCount > delegate.expandedGroupChildrenCount || delegate.isGroupExpanded)
                                    && delegate.ListView.nextSection !== delegate.ListView.section
                                onClicked: list.setGroupExpanded(delegate.index, !delegate.isGroupExpanded)
                            }
                        }
                    }
                }
            }

            Loader {
                anchors.centerIn: parent
                width: parent.width - (Kirigami.Units.gridUnit * 4)

                active: list.count === 0
                visible: active
                asynchronous: true

                sourceComponent: NotificationManager.Server.valid ? noUnreadMessage : notAvailableMessage
            }

            Component {
                id: noUnreadMessage

                PlasmaExtras.PlaceholderMessage {
                    anchors.centerIn: parent
                    width: parent.width

                    iconName: "checkmark"
                    text: i18n("No unread notifications")
                }
            }

            Component {
                id: notAvailableMessage

                PlasmaExtras.PlaceholderMessage {
                    // Checking valid to avoid creating ServerInfo object if everything is alright
                    readonly property NotificationManager.ServerInfo currentOwner: NotificationManager.Server.currentOwner

                    anchors.centerIn: parent
                    width: parent.width

                    iconName: "notifications-disabled"
                    text: i18n("Notification service not available")
                    explanation: currentOwner && currentOwner.vendor && currentOwner.name
                                ? i18nc("Vendor and product name", "Notifications are currently provided by '%1 %2'", currentOwner.vendor, currentOwner.name)
                                : ""
                }
            }
        }
    }
}
