/*
 * Copyright © 2017 Timothy Arceri
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/**
 * \file active-sampler-conflict.c
 *
 * Verify results of program validation when a conflicting sampler
 * configuration is used via the binding layout qualifier.
 *
 * Section 11.1.3.11 (Validation) of the OpenGL 4.5 spec says:
 *
 *     "An INVALID_OPERATION error is generated by any command that trans-
 *     fers vertices to the GL or launches compute work if the current set of
 *     active program objects cannot be executed, for reasons including:
 *
 *         - any two active samplers in the current program object are
 *           of different types, but refer to the same texture image
 *           unit,"
 *
 * Here we test the INVALID_OPERATION by both calling glValidateProgram
 * and glDrawArrays.
 */
#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_core_version = 31;

	config.khr_no_error_support = PIGLIT_HAS_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

static const char *vs_code_binding_0 =
	"#version 140\n"
	"#extension GL_ARB_shading_language_420pack: require\n"
	"layout(binding = 0) uniform samplerBuffer buffer;\n"
	"\n"
	"void main()\n"
	"{\n"
	"    gl_Position = texelFetch(buffer, 0);\n"
	"}\n"
	;

static const char *vs_code_binding_1 =
	"#version 140\n"
	"#extension GL_ARB_shading_language_420pack: require\n"
	"layout(binding = 1) uniform samplerBuffer buffer;\n"
	"\n"
	"void main()\n"
	"{\n"
	"    gl_Position = texelFetch(buffer, 0);\n"
	"}\n"
	;

static const char *fs_code_binding_0 =
	"#version 140\n"
	"#extension GL_ARB_shading_language_420pack: require\n"
	"layout(binding = 0) uniform samplerCube cubemapTex;\n"
	"\n"
	"void main()\n"
	"{\n"
	"    gl_FragColor = textureCube(cubemapTex, vec3(gl_FragCoord.xyz));\n"
	"}\n"
	;

static bool
program_check_status(GLuint prog)
{
        GLchar *info = NULL;
        GLint size;
        GLint ok;

        glValidateProgram(prog);
        glGetProgramiv(prog, GL_VALIDATE_STATUS, &ok);

        /* Some drivers return a size of 1 for an empty log.  This is the size
         * of a log that contains only a terminating NUL character.
         */
        glGetProgramiv(prog, GL_INFO_LOG_LENGTH, &size);
        if (size > 1) {
                info = malloc(size);
                glGetProgramInfoLog(prog, size, NULL, info);
        }

        if (!ok) {
                printf("Failed to validate the program: %s\n",
		       (info != NULL) ? info : "<empty log>");
        }
        else if (0 && info != NULL) {
                /* Enable this to get extra linking info.
                 * Even if there's no link errors, the info log may
                 * have some remarks.
                 */
                printf("Program validation warning: %s\n", info);
        }

        free(info);

        return ok;
}


void piglit_init(int argc, char **argv)
{
	GLuint prog;
	bool pass = true;

	piglit_require_extension("GL_ARB_shading_language_420pack");

	GLuint vao;
	glGenVertexArrays(1, &vao);
	glBindVertexArray(vao);

	/* First, try an invalid combination */
	prog = piglit_build_simple_program(vs_code_binding_0,
					   fs_code_binding_0);
	glUseProgram(prog);

	if (program_check_status(prog)) {
		fprintf(stderr,
			"Program was validated with conflicting "
			"sampler configuration.\n");
		pass = false;
	}

	glDrawArrays(GL_TRIANGLE_STRIP, 0, 3);

	pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;

	/* Re-validate the invalid program. */
	if (program_check_status(prog)) {
		fprintf(stderr,
			"Program was validated with conflicting "
			"sampler configuration (second attempt).\n");
		pass = false;
	}

	/* Clean up. */
	glUseProgram(0);
	glDeleteProgram(prog);

	/* Now try a valid combination. */
	prog = piglit_build_simple_program(vs_code_binding_1,
					   fs_code_binding_0);
	glUseProgram(prog);

	if (!program_check_status(prog)) {
		fprintf(stderr,
			"Program was not validated with non-conflicting "
			"sampler configuration.\n");
		pass = false;
	}


	glDrawArrays(GL_TRIANGLE_STRIP, 0, 3);

	pass = piglit_check_gl_error(GL_NO_ERROR) && pass;

	/* Re-validate the valid program. */
	if (!program_check_status(prog)) {
		fprintf(stderr,
			"Program was not validated with non-conflicting "
			"sampler configuration (second attempt).\n");
		pass = false;
	}

	/* Clean up. */
	glUseProgram(0);
	glDeleteProgram(prog);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result
piglit_display(void)
{
	/* Not reached */
	return PIGLIT_FAIL;
}
