/*  
 *  PCSX2 Dev Team
 *  Copyright (C) 2015
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

#include "GamePad.h"
#ifdef SDL_BUILD
#include "SDL/joystick.h"
#endif

std::vector<std::unique_ptr<GamePad>> s_vgamePad;

/**
 * Following static methods are just forwarders to their backend
 * This is where link between agnostic and specific code is done
 **/

/**
 * Find every interesting devices and create right structure for them(depend on backend)
 **/
void GamePad::EnumerateGamePads(std::vector<std::unique_ptr<GamePad>> &vgamePad)
{
#ifdef SDL_BUILD
    JoystickInfo::EnumerateJoysticks(vgamePad);
#endif
}

/**
 * Safely dispatch to the Rumble method above
 **/
void GamePad::DoRumble(unsigned type, unsigned pad)
{
    int index = uid_to_index(pad);
    if (index >= 0)
        s_vgamePad[index]->Rumble(type, pad);
}

size_t GamePad::index_to_uid(int index)
{
    if ((index >= 0) && (index < (int)s_vgamePad.size()))
        return s_vgamePad[index]->GetUniqueIdentifier();
    else
        return 0;
}

int GamePad::uid_to_index(int pad)
{
    size_t uid = g_conf.get_joy_uid(pad);

    for (int i = 0; i < (int)s_vgamePad.size(); ++i) {
        if (s_vgamePad[i]->GetUniqueIdentifier() == uid)
            return i;
    }

    // Current uid wasn't found maybe the pad was unplugged. Or
    // user didn't select it. Fallback to 1st pad for
    // 1st player. And 2nd pad for 2nd player.
    if ((int)s_vgamePad.size() > pad)
        return pad;

    return -1;
}
