#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

from fenrirscreenreader.core import debug
from fenrirscreenreader.core.i18n import _
from fenrirscreenreader.core.settingsData import settings_data


class QuickMenuManager:
    def __init__(self):
        self.position = 0
        self.quickMenu = []
        self.settings = settings_data

    def initialize(self, environment):
        self.env = environment
        self.load_menu(
            self.env["runtime"]["SettingsManager"].get_setting(
                "menu", "quickMenu"
            )
        )

    def shutdown(self):
        pass

    def load_menu(self, menuString):
        self.position = 0
        self.quickMenu = []
        if menuString == "":
            return
        entrys = menuString.split(";")
        for e in entrys:
            entry = e.split("#")
            if len(entry) != 2:
                continue
            try:
                t = self.settings[entry[0]][entry[1]]
            except Exception as e:
                self.env["runtime"]["DebugManager"].write_debug_out(
                    "QuickMenuManager load_menu: Setting not found "
                    + str(entry)
                    + ": "
                    + str(e),
                    debug.DebugLevel.ERROR,
                )
                print(entry[0], entry[1], "not found")
                continue
            self.quickMenu.append({"section": entry[0], "setting": entry[1]})

    def next_entry(self):
        if len(self.quickMenu) == 0:
            return False
        self.position += 1
        if self.position >= len(self.quickMenu):
            self.position = 0
        return True

    def prev_entry(self):
        if len(self.quickMenu) == 0:
            return False
        self.position -= 1
        if self.position < 0:
            self.position = len(self.quickMenu) - 1
        return True

    def next_value(self):
        if len(self.quickMenu) == 0:
            return False
        section = self.quickMenu[self.position]["section"]
        setting = self.quickMenu[self.position]["setting"]
        value_string = ""
        try:
            value_string = self.env["runtime"]["SettingsManager"].get_setting(
                section, setting
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "QuickMenuManager next_value: Error getting setting value: "
                + str(e),
                debug.DebugLevel.ERROR,
            )
            return False

        try:
            if isinstance(self.settings[section][setting], str):
                value = str(value_string)
                return False
            elif isinstance(self.settings[section][setting], bool):
                if value_string not in ["True", "False"]:
                    return False
                value = not value
                self.env["runtime"]["SettingsManager"].set_setting(
                    section, setting, str(value)
                )
            elif isinstance(self.settings[section][setting], int):
                value = int(value_string)
                value += 1
                self.env["runtime"]["SettingsManager"].set_setting(
                    section, setting, str(value)
                )
            elif isinstance(self.settings[section][setting], float):
                value = float(value_string)
                value += 0.05
                if value > 1.0:
                    value = 1.0
                self.env["runtime"]["SettingsManager"].set_setting(
                    section, setting, str(value)[:4]
                )
        except Exception as e:
            return False
        return True

    def prev_value(self):
        if len(self.quickMenu) == 0:
            return False
        section = self.quickMenu[self.position]["section"]
        setting = self.quickMenu[self.position]["setting"]
        value_string = ""
        try:
            value_string = self.env["runtime"]["SettingsManager"].get_setting(
                section, setting
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "QuickMenuManager prev_value: Error getting setting value: "
                + str(e),
                debug.DebugLevel.ERROR,
            )
            return False
        try:
            if isinstance(self.settings[section][setting], str):
                value = str(value_string)
                return False
            elif isinstance(self.settings[section][setting], bool):
                if value_string not in ["True", "False"]:
                    return False
                value = not value
                self.env["runtime"]["SettingsManager"].set_setting(
                    section, setting, str(value)
                )
            elif isinstance(self.settings[section][setting], int):
                value = int(value_string)
                value -= 1
                if value < 0:
                    value = 0
                self.env["runtime"]["SettingsManager"].set_setting(
                    section, setting, str(value)
                )
            elif isinstance(self.settings[section][setting], float):
                value = float(value_string)
                value -= 0.05
                if value < 0.0:
                    value = 0.0
                self.env["runtime"]["SettingsManager"].set_setting(
                    section, setting, str(value)[:4]
                )
        except Exception as e:
            return False
        return True

    def get_current_entry(self):
        if len(self.quickMenu) == 0:
            return ""
        try:
            return (
                _(self.quickMenu[self.position]["section"])
                + " "
                + _(self.quickMenu[self.position]["setting"])
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "QuickMenuManager get_current_entry: Error formatting entry: "
                + str(e),
                debug.DebugLevel.ERROR,
            )
            return _("setting invalid")

    def get_current_value(self):
        if len(self.quickMenu) == 0:
            return ""
        try:
            return self.env["runtime"]["SettingsManager"].get_setting(
                self.quickMenu[self.position]["section"],
                self.quickMenu[self.position]["setting"],
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "QuickMenuManager get_current_value: Error getting setting value: "
                + str(e),
                debug.DebugLevel.ERROR,
            )
            return _("setting value invalid")
