#!/usr/bin/env python3

import importlib.util
import os

from fenrirscreenreader.core.i18n import _

# Load base configuration class
_base_path = os.path.join(os.path.dirname(__file__), "..", "config_base.py")
_spec = importlib.util.spec_from_file_location("config_base", _base_path)
_module = importlib.util.module_from_spec(_spec)
_spec.loader.exec_module(_module)
config_command = _module.config_command


class command(config_command):
    def __init__(self):
        super().__init__()

    def get_description(self):
        return "Select keyboard layout (desktop or laptop)"

    def run(self):
        current_layout = self.get_setting(
            "keyboard", "keyboardLayout", "desktop"
        )

        # Present current layout
        self.present_text(f"Current keyboard layout: {current_layout}")

        # Find available keyboard layouts
        keyboard_path = "/etc/fenrirscreenreader/keyboard"
        if not os.path.isdir(keyboard_path):
            # Development path
            keyboard_path = os.path.join(
                os.path.dirname(self.settings_file), "..", "keyboard"
            )

        available_layouts = self.get_available_layouts(keyboard_path)

        if len(available_layouts) > 1:
            # Cycle through available layouts
            try:
                current_index = available_layouts.index(current_layout)
                next_index = (current_index + 1) % len(available_layouts)
                new_layout = available_layouts[next_index]
            except ValueError:
                # Current layout not found, use first available
                new_layout = available_layouts[0]

            success = self.set_setting(
                "keyboard", "keyboardLayout", new_layout
            )

            if success:
                self.present_text(f"Keyboard layout changed to: {new_layout}")
                self.present_text(
                    "Please restart Fenrir for this change to take effect."
                )
                self.play_sound("Accept")
            else:
                self.present_text("Failed to change keyboard layout")
                self.play_sound("Error")
        else:
            self.present_text("Only default keyboard layout is available")
            self.play_sound("Cancel")

    def get_available_layouts(self, keyboard_path):
        """Find available keyboard layout files"""
        layouts = []

        if os.path.isdir(keyboard_path):
            try:
                for file in os.listdir(keyboard_path):
                    if file.endswith(".conf") and not file.startswith("."):
                        layout_name = file[:-5]  # Remove .conf extension
                        layouts.append(layout_name)
            except Exception:
                pass

        # Ensure we have at least the default layouts
        if not layouts:
            layouts = ["desktop", "laptop"]

        return sorted(layouts)
